/*
 * Decompiled with CFR 0.152.
 */
package jadx.api.plugins.utils;

import jadx.api.plugins.utils.CommonFileUtils;
import jadx.api.plugins.utils.LimitedInputStream;
import jadx.core.utils.Utils;
import jadx.core.utils.exceptions.JadxRuntimeException;
import jadx.core.utils.files.ZipFile;
import java.io.BufferedInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.Enumeration;
import java.util.function.BiConsumer;
import java.util.function.BiFunction;
import java.util.zip.ZipEntry;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ZipSecurity {
    private static final Logger LOG = LoggerFactory.getLogger(ZipSecurity.class);
    private static final boolean DISABLE_CHECKS = Utils.getEnvVarBool("JADX_DISABLE_ZIP_SECURITY", false);
    private static final int ZIP_BOMB_DETECTION_FACTOR = 100;
    private static final int ZIP_BOMB_MIN_UNCOMPRESSED_SIZE = 0x1900000;
    private static final int MAX_ENTRIES_COUNT = Utils.getEnvVarInt("JADX_ZIP_MAX_ENTRIES_COUNT", 100000);

    private ZipSecurity() {
    }

    private static boolean isInSubDirectoryInternal(File baseDir, File file) {
        File current = file;
        while (current != null) {
            if (current.equals(baseDir)) {
                return true;
            }
            current = current.getParentFile();
        }
        return false;
    }

    public static boolean isInSubDirectory(File baseDir, File file) {
        if (DISABLE_CHECKS) {
            return true;
        }
        try {
            return ZipSecurity.isInSubDirectoryInternal(baseDir.getCanonicalFile(), file.getCanonicalFile());
        }
        catch (IOException e) {
            return false;
        }
    }

    public static boolean isValidZipEntryName(String entryName) {
        if (DISABLE_CHECKS) {
            return true;
        }
        if (entryName.contains("..") && (entryName.contains("../") || entryName.contains("..\\"))) {
            LOG.error("Path traversal attack detected in entry: '{}'", (Object)entryName);
            return false;
        }
        try {
            File currentPath = CommonFileUtils.CWD;
            File canonical = new File(currentPath, entryName).getCanonicalFile();
            if (ZipSecurity.isInSubDirectoryInternal(currentPath, canonical)) {
                return true;
            }
        }
        catch (Exception exception) {
            // empty catch block
        }
        LOG.error("Invalid file name or path traversal attack detected: {}", (Object)entryName);
        return false;
    }

    public static boolean isZipBomb(ZipEntry entry) {
        boolean possibleZipBomb;
        if (DISABLE_CHECKS) {
            return false;
        }
        long compressedSize = entry.getCompressedSize();
        long uncompressedSize = entry.getSize();
        boolean invalidSize = compressedSize < 0L || uncompressedSize < 0L;
        boolean bl = possibleZipBomb = uncompressedSize >= 0x1900000L && compressedSize * 100L < uncompressedSize;
        if (invalidSize || possibleZipBomb) {
            LOG.error("Potential zip bomb attack detected, invalid sizes: compressed {}, uncompressed {}, name {}", new Object[]{compressedSize, uncompressedSize, entry.getName()});
            return true;
        }
        return false;
    }

    public static boolean isValidZipEntry(ZipEntry entry) {
        return ZipSecurity.isValidZipEntryName(entry.getName()) && !ZipSecurity.isZipBomb(entry);
    }

    public static InputStream getInputStreamForEntry(ZipFile zipFile, ZipEntry entry) throws IOException {
        if (DISABLE_CHECKS) {
            return new BufferedInputStream(zipFile.getInputStream(entry));
        }
        InputStream in = zipFile.getInputStream(entry);
        LimitedInputStream limited = new LimitedInputStream(in, entry.getSize());
        return new BufferedInputStream(limited);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Nullable
    public static <R> R visitZipEntries(File file, BiFunction<ZipFile, ZipEntry, R> visitor) {
        try (ZipFile zip = new ZipFile(file);){
            Enumeration<? extends ZipEntry> entries = zip.entries();
            int entriesProcessed = 0;
            while (entries.hasMoreElements()) {
                ZipEntry entry = entries.nextElement();
                if (!ZipSecurity.isValidZipEntry(entry)) continue;
                R result = visitor.apply(zip, entry);
                if (result != null) {
                    R r = result;
                    return r;
                }
                if (!DISABLE_CHECKS && ++entriesProcessed > MAX_ENTRIES_COUNT) throw new JadxRuntimeException("Zip entries count limit exceeded: " + MAX_ENTRIES_COUNT + ", last entry: " + entry.getName());
            }
            return null;
        }
        catch (Exception e) {
            throw new JadxRuntimeException("Failed to process zip file: " + file.getAbsolutePath(), e);
        }
    }

    public static void readZipEntries(File file, BiConsumer<ZipEntry, InputStream> visitor) {
        ZipSecurity.visitZipEntries(file, (zip, entry) -> {
            if (!entry.isDirectory()) {
                try (InputStream in = ZipSecurity.getInputStreamForEntry(zip, entry);){
                    visitor.accept((ZipEntry)entry, in);
                }
                catch (Exception e) {
                    throw new JadxRuntimeException("Failed to process zip entry: " + entry.getName());
                }
            }
            return null;
        });
    }
}

