import { Sequencer } from "../../../spessasynth_lib/sequencer/sequencer.js";
import { formatTime } from "../../../spessasynth_lib/utils/other.js";
import { supportedEncodings } from "../utils/encodings.js";
import { getBackwardSvg, getForwardSvg, getLoopSvg, getPauseSvg, getPlaySvg, getTextSvg } from "../utils/icons.js";
import { messageTypes } from "../../../spessasynth_lib/midi_parser/midi_message.js";
import { getSeqUIButton } from "./sequi_button.js";
import { keybinds } from "../utils/keybinds.js";
import { createNavigatorHandler, updateTitleAndMediaStatus } from "./title_and_media_status.js";
import { createLyrics, setLyricsText, updateOtherTextEvents } from "./lyrics.js";
import { RMIDINFOChunks } from "../../../spessasynth_lib/midi_parser/rmidi_writer.js";

/**
 * sequencer_ui.js
 * purpose: manages the GUI for the sequencer class, adding buttons for pause/play, lyrics, next, previous etc.
 */

const ICON_SIZE = 32;

const ICON_COLOR = "#ccc";
const ICON_DISABLED_COLOR = "#555";

const ICON_COLOR_L = "#333";
const ICON_DISABLED_COLOR_L = "#ddd";

const DEFAULT_ENCODING = "Shift_JIS";

class SequencerUI
{
    /**
     * Creates a new User Interface for the given MidiSequencer
     * @param element {HTMLElement} the element to create sequi in
     * @param locale {LocaleManager}
     * @param musicMode {MusicModeUI}
     * @param renderer {Renderer}
     */
    constructor(element, locale, musicMode, renderer)
    {
        this.iconColor = ICON_COLOR;
        this.iconDisabledColor = ICON_DISABLED_COLOR;
        this.controls = element;
        this.encoding = DEFAULT_ENCODING;
        this.decoder = new TextDecoder(this.encoding);
        this.infoDecoder = new TextDecoder(this.encoding);
        this.hasInfoDecoding = false;
        /**
         * the currently displayed (highlighted) lyrics event index
         * @type {number}
         */
        this.lyricsIndex = 0;
        this.requiresTextUpdate = false;
        /**
         * @type {{type: messageTypes, data: Uint8Array}[]}
         */
        this.rawOtherTextEvents = [];
        this.mode = "dark";
        this.locale = locale;
        this.currentSongTitle = "";
        /**
         * @type {Uint8Array[]}
         */
        this.currentLyrics = [];
        /**
         * Current lyrics decoded as strings
         * @type {string[]}
         */
        this.currentLyricsString = [];
        this.musicModeUI = musicMode;
        this.renderer = renderer;
        
        this.mainTitleMessageDisplay = document.getElementById("title");
        this.synthDisplayMode = {
            enabled: false,
            currentEncodedText: new Uint8Array(0)
        };
        
        // set up synth display event
        let displayTimeoutId = null;
        renderer.synth.eventHandler.addEvent("synthdisplay", "sequi-synth-display", data =>
        {
            if (data.displayType === 0 || data.displayType === 1)
            {
                // clear styles and apply monospace
                this.mainTitleMessageDisplay.classList.add("sysex_display");
                this.mainTitleMessageDisplay.classList.remove("xg_sysex_display");
                let textData = data.displayData;
                // remove "Display Letters" byte before decoding for XG display
                if (data.displayType === 1)
                {
                    textData = textData.slice(1);
                }
                // decode the text
                let text = this.decodeTextFix(textData.buffer);
                
                // XG is type 1, apply some fixes to it.
                // XG Displays have a special behavior, we try to mimic it here
                // reference video:
                // https://www.youtube.com/watch?v=_mR7DV1E4KE
                // first of all, extract the "Display Letters" byte
                if (data.displayType === 1)
                {
                    const displayLetters = data.displayData[0];
                    // XG Display Letters:
                    // the screen is monospace,
                    // two rows, 16 characters each (max)
                    // since this is XG data, apply the XG display style
                    this.mainTitleMessageDisplay.classList.add("xg_sysex_display");
                    
                    // 0x0c where c are the amount of spaces prepended
                    const spaces = displayLetters & 0x0F;
                    for (let i = 0; i < spaces; i++)
                    {
                        text = " " + text;
                    }
                    
                    // at 16 characters, add a newline
                    if (text.length >= 16)
                    {
                        text = text.slice(0, 16) + "\n" + text.slice(16);
                    }
                    
                    // if type is 0x1x, add a newline
                    if ((displayLetters & 0x10) > 1)
                    {
                        text = "\n" + text;
                    }
                    
                }
                
                
                if (text.trim().length === 0)
                {
                    // set the text to invisible character to keep the height
                    this.mainTitleMessageDisplay.innerText = "‎ ";
                }
                else
                {
                    // set the display to an invisible character to keep the height
                    this.mainTitleMessageDisplay.innerText = text;
                }
                
                this.synthDisplayMode.enabled = true;
                this.synthDisplayMode.currentEncodedText = textData;
                if (displayTimeoutId !== null)
                {
                    clearTimeout(displayTimeoutId);
                }
                displayTimeoutId = setTimeout(() =>
                {
                    this.synthDisplayMode.enabled = false;
                    this.restoreDisplay();
                }, 5000);
                
            }
        });
    }
    
    toggleDarkMode()
    {
        if (this.mode === "dark")
        {
            this.mode = "light";
            this.iconColor = ICON_COLOR_L;
            this.iconDisabledColor = ICON_DISABLED_COLOR_L;
        }
        else
        {
            this.mode = "dark";
            this.iconColor = ICON_COLOR;
            this.iconDisabledColor = ICON_DISABLED_COLOR;
        }
        if (!this.seq)
        {
            this.requiresThemeUpdate = true;
            return;
        }
        this.progressBar.classList.toggle("note_progress_light");
        this.progressBarBackground.classList.toggle("note_progress_background_light");
        this.lyricsElement.mainDiv.classList.toggle("lyrics_light");
        this.lyricsElement.titleWrapper.classList.toggle("lyrics_light");
        this.lyricsElement.selector.classList.toggle("lyrics_light");
    }
    
    seqPlay(sendPlay = true)
    {
        if (sendPlay)
        {
            this.seq.play();
        }
        this.playPause.innerHTML = getPauseSvg(ICON_SIZE);
        this.createNavigatorHandler();
        this.updateTitleAndMediaStatus();
        if (!navigator.mediaSession)
        {
            return;
        }
        navigator.mediaSession.playbackState = "playing";
    }
    
    seqPause(sendPause = true)
    {
        if (sendPause)
        {
            this.seq.pause();
        }
        this.playPause.innerHTML = getPlaySvg(ICON_SIZE);
        this.createNavigatorHandler();
        this.updateTitleAndMediaStatus();
        if (!navigator.mediaSession)
        {
            return;
        }
        navigator.mediaSession.playbackState = "paused";
    }
    
    switchToNextSong()
    {
        this.seq.nextSong();
        this.createNavigatorHandler();
        this.updateTitleAndMediaStatus();
    }
    
    switchToPreviousSong()
    {
        this.seq.previousSong();
        this.createNavigatorHandler();
        this.updateTitleAndMediaStatus();
    }
    
    /**
     * @param text {ArrayBuffer}
     * @returns {string}
     */
    decodeTextFix(text)
    {
        let encodingIndex = 0;
        while (true)
        {
            try
            {
                return this.decoder.decode(text);
            }
            catch (e)
            {
                encodingIndex++;
                this.changeEncoding(supportedEncodings[encodingIndex]);
                this.encodingSelector.value = supportedEncodings[encodingIndex];
            }
        }
    }
    
    /**
     *
     * @param sequencer {Sequencer} the sequencer to be used
     */
    connectSequencer(sequencer)
    {
        this.seq = sequencer;
        this.createControls();
        this.setSliderInterval();
        this.createNavigatorHandler();
        this.updateTitleAndMediaStatus();
        
        this.seq.onTextEvent = (data, type, lyricsIndex) =>
        {
            switch (type)
            {
                default:
                    return;
                
                case messageTypes.text:
                case messageTypes.copyright:
                case messageTypes.cuePoint:
                case messageTypes.trackName:
                case messageTypes.instrumentName:
                case messageTypes.programName:
                case messageTypes.marker:
                    this.rawOtherTextEvents.push({ type: type, data: data });
                    this.requiresTextUpdate = true;
                    return;
                
                case messageTypes.lyric:
                    this.setLyricsText(lyricsIndex);
                    break;
            }
        };
        
        this.seq.addOnTimeChangeEvent(() =>
        {
            this.seqPlay(false);
        }, "sequi-time-change");
        
        this.seq.addOnSongChangeEvent(data =>
        {
            this.synthDisplayMode.enabled = false;
            this.lyricsIndex = 0;
            this.createNavigatorHandler();
            this.updateTitleAndMediaStatus();
            this.seqPlay(false);
            // disable loop if more than 1 song
            if (this.seq.songsAmount > 1)
            {
                this.seq.loop = false;
                this.loopButton.firstElementChild.setAttribute(
                    "fill",
                    this.iconDisabledColor
                );
            }
            this.restoreDisplay();
            
            // use encoding suggested by the rmidi if available
            this.hasInfoDecoding = this.seq.midiData.RMIDInfo?.[RMIDINFOChunks.encoding] !== undefined;
            if (data.isEmbedded)
            {
                /**
                 * @param type {string}
                 * @param def {string}
                 * @param decoder {TextDecoder}
                 * @param prepend {string}
                 * @return {string}
                 */
                const verifyDecode = (type, def, decoder, prepend = "") =>
                {
                    return this.seq.midiData.RMIDInfo?.[type] === undefined ? def : prepend + decoder.decode(
                        this.seq.midiData.RMIDInfo?.[type]).replace(/\0$/, "");
                };
                const dec = new TextDecoder();
                const midiEncoding = verifyDecode(
                    RMIDINFOChunks.midiEncoding,
                    this.encoding,
                    dec
                );
                const infoEncoding = verifyDecode(RMIDINFOChunks.encoding, "utf-8", dec);
                this.infoDecoder = new TextDecoder(infoEncoding);
                this.changeEncoding(midiEncoding);
            }
        }, "sequi-song-change");
        
        if (this.requiresThemeUpdate)
        {
            if (this.mode === "light")
            {
                // change to dark and then switch
                this.mode = "dark";
                this.toggleDarkMode();
            }
            // otherwise we're already dark
        }
    }
    
    /**
     * Restores the display to the current song title and removes the SysEx display styles
     */
    restoreDisplay()
    {
        let textToShow = this.currentSongTitle;
        if (!this.seq)
        {
            // set to default title
            textToShow = this.locale.getLocaleString("locale.titleMessage");
        }
        this.mainTitleMessageDisplay.innerText = textToShow;
        this.mainTitleMessageDisplay.classList.remove("sysex_display");
        this.mainTitleMessageDisplay.classList.remove("xg_sysex_display");
    }
    
    changeEncoding(encoding)
    {
        this.encoding = encoding;
        this.decoder = new TextDecoder(encoding);
        if (!this.hasInfoDecoding)
        {
            this.infoDecoder = new TextDecoder(encoding);
        }
        this.updateOtherTextEvents();
        // update all spans with the new encoding
        this.lyricsElement.text.separateLyrics.forEach((span, index) =>
        {
            if (this.currentLyrics[index] === undefined)
            {
                return;
            }
            span.innerText = this.decodeTextFix(this.currentLyrics[index].buffer);
        });
        this.lyricsElement.selector.value = encoding;
        this.updateTitleAndMediaStatus(false);
        this.setLyricsText(this.lyricsIndex);
    }
    
    createControls()
    {
        // time
        this.progressTime = document.createElement("p");
        this.progressTime.id = "note_time";
        // it'll always be on top
        this.progressTime.onclick = event =>
        {
            event.preventDefault();
            const barPosition = progressBarBg.getBoundingClientRect();
            const x = event.clientX - barPosition.left;
            const width = barPosition.width;
            
            this.seq.currentTime = (x / width) * this.seq.duration;
            playPauseButton.innerHTML = getPauseSvg(ICON_SIZE);
        };
        
        this.createLyrics();
        
        
        // background bar
        const progressBarBg = document.createElement("div");
        progressBarBg.id = "note_progress_background";
        this.progressBarBackground = progressBarBg;
        
        
        // foreground bar
        this.progressBar = document.createElement("div");
        this.progressBar.id = "note_progress";
        this.progressBar.min = (0).toString();
        this.progressBar.max = this.seq.duration.toString();
        
        
        // control buttons
        const controlsDiv = document.createElement("div");
        
        
        // play pause
        const playPauseButton = getSeqUIButton(
            "Play/Pause",
            getPauseSvg(ICON_SIZE)
        );
        this.playPause = playPauseButton;
        this.locale.bindObjectProperty(playPauseButton, "title", "locale.sequencerController.playPause");
        const togglePlayback = () =>
        {
            if (this.seq.paused)
            {
                this.seqPlay();
            }
            else
            {
                this.seqPause();
            }
        };
        playPauseButton.onclick = togglePlayback;
        
        
        // previous song button
        const previousSongButton = getSeqUIButton(
            "Previous song",
            getBackwardSvg(ICON_SIZE)
        );
        this.locale.bindObjectProperty(previousSongButton, "title", "locale.sequencerController.previousSong");
        previousSongButton.onclick = () => this.switchToPreviousSong();
        
        // next song button
        const nextSongButton = getSeqUIButton(
            "Next song",
            getForwardSvg(ICON_SIZE)
        );
        this.locale.bindObjectProperty(nextSongButton, "title", "locale.sequencerController.nextSong");
        nextSongButton.onclick = () => this.switchToNextSong();
        
        // loop button
        const loopButton = getSeqUIButton(
            "Loop this",
            getLoopSvg(ICON_SIZE)
        );
        this.locale.bindObjectProperty(loopButton, "title", "locale.sequencerController.loopThis");
        const toggleLoop = () =>
        {
            if (this.seq.loop)
            {
                this.seq.loop = false;
            }
            else
            {
                this.seq.loop = true;
                if (this.seq.currentTime >= this.seq.duration)
                {
                    this.seq.currentTime = 0;
                }
            }
            loopButton.firstElementChild.setAttribute(
                "fill",
                (this.seq.loop ? this.iconColor : this.iconDisabledColor)
            );
        };
        loopButton.onclick = toggleLoop;
        this.loopButton = loopButton;
        
        
        // show text button
        const textButton = getSeqUIButton(
            "Show lyrics",
            getTextSvg(ICON_SIZE)
        );
        this.locale.bindObjectProperty(textButton, "title", "locale.sequencerController.lyrics.show");
        textButton.firstElementChild.setAttribute("fill", this.iconDisabledColor); // defaults to disabled
        const toggleLyrics = () =>
        {
            this.lyricsElement.mainDiv.classList.toggle("lyrics_show");
            textButton.firstElementChild.setAttribute(
                "fill",
                (this.lyricsElement.mainDiv.classList.contains("lyrics_show") ? this.iconColor : this.iconDisabledColor)
            );
        };
        this.toggleLyrics = toggleLyrics;
        textButton.onclick = toggleLyrics;
        
        // keyboard control
        document.addEventListener("keydown", event =>
        {
            switch (event.key.toLowerCase())
            {
                case keybinds.playPause:
                    event.preventDefault();
                    togglePlayback();
                    break;
                
                case keybinds.toggleLoop:
                    event.preventDefault();
                    toggleLoop();
                    break;
                
                case keybinds.toggleLyrics:
                    event.preventDefault();
                    toggleLyrics();
                    break;
                
                default:
                    break;
            }
        });
        
        // add everything
        controlsDiv.appendChild(previousSongButton); // |<
        controlsDiv.appendChild(loopButton);         // ()
        controlsDiv.appendChild(playPauseButton);    // ||
        controlsDiv.appendChild(textButton);         // ==
        controlsDiv.appendChild(nextSongButton);     // >|
        
        this.controls.appendChild(progressBarBg);
        progressBarBg.appendChild(this.progressBar);
        this.controls.appendChild(this.progressTime);
        this.controls.appendChild(controlsDiv);
        
        // add number and arrow controls
        document.addEventListener("keydown", e =>
        {
            
            switch (e.key.toLowerCase())
            {
                case keybinds.seekBackwards:
                    e.preventDefault();
                    this.seq.currentTime -= 5;
                    playPauseButton.innerHTML = getPauseSvg(ICON_SIZE);
                    break;
                
                case keybinds.seekForwards:
                    e.preventDefault();
                    this.seq.currentTime += 5;
                    playPauseButton.innerHTML = getPauseSvg(ICON_SIZE);
                    break;
                
                case keybinds.previousSong:
                    this.switchToPreviousSong();
                    break;
                
                case keybinds.nextSong:
                    this.switchToNextSong();
                    break;
                
                default:
                    if (!isNaN(parseFloat(e.key)))
                    {
                        e.preventDefault();
                        const num = parseInt(e.key);
                        if (0 <= num && num <= 9)
                        {
                            this.seq.currentTime = this.seq.duration * (num / 10);
                            playPauseButton.innerHTML = getPauseSvg(ICON_SIZE);
                        }
                    }
                    break;
            }
        });
    }
    
    _updateInterval()
    {
        this.progressBar.style.width = `${(this.seq.currentTime / this.seq.duration) * 100}%`;
        const time = formatTime(this.seq.currentTime);
        const total = formatTime(this.seq.duration);
        this.progressTime.innerText = `${time.time} / ${total.time}`;
        if (this.requiresTextUpdate)
        {
            this.updateOtherTextEvents();
            this.requiresTextUpdate = false;
        }
    }
    
    setSliderInterval()
    {
        setInterval(this._updateInterval.bind(this), 100);
    }
    
    loadLyricData()
    {
        // load lyrics
        this.currentLyrics = this.seq.midiData.lyrics;
        this.currentLyricsString = this.currentLyrics.map(l => this.decodeTextFix(l.buffer));
        if (this.currentLyrics.length === 0)
        {
            this.currentLyricsString = [this.locale.getLocaleString("locale.sequencerController.lyrics.noLyrics")];
        }
        else
        {
            // perform a check for double lyrics:
            // for example in some midi's, every lyric event is duplicated:
            // "He's " turns into two "He's " and another "He's " event
            // if that's the case for all events in the current lyrics, set duplicates to "" to avoid index errors
            let isDoubleLyrics = true;
            // note: the first lyrics is usually a control character
            for (let i = 1; i < this.currentLyricsString.length - 1; i += 2)
            {
                const first = this.currentLyricsString[i];
                const second = this.currentLyricsString[i + 1];
                // note: newline should be skipped
                if (first.trim() === "" || second.trim() === "")
                {
                    i -= 1;
                    continue;
                }
                if (first.trim() !== second.trim())
                {
                    isDoubleLyrics = false;
                    break;
                }
            }
            if (isDoubleLyrics)
            {
                for (let i = 0; i < this.currentLyricsString.length; i += 2)
                {
                    // note: newline should be skipped
                    if (this.currentLyricsString[i] === "\n")
                    {
                        i -= 1;
                        continue;
                    }
                    this.currentLyricsString[i] = "";
                }
                
            }
            
        }
        // create lyrics as separate spans
        // clear previous lyrics
        this.lyricsElement.text.main.innerHTML = "";
        this.lyricsElement.text.separateLyrics = [];
        for (const lyric of this.currentLyricsString)
        {
            const span = document.createElement("span");
            span.innerText = lyric;
            // gray (not highlighted) text
            span.classList.add("lyrics_text_gray");
            this.lyricsElement.text.main.appendChild(span);
            this.lyricsElement.text.separateLyrics.push(span);
        }
        
    }
}

SequencerUI.prototype.createNavigatorHandler = createNavigatorHandler;
SequencerUI.prototype.updateTitleAndMediaStatus = updateTitleAndMediaStatus;

SequencerUI.prototype.createLyrics = createLyrics;
SequencerUI.prototype.setLyricsText = setLyricsText;
SequencerUI.prototype.updateOtherTextEvents = updateOtherTextEvents;

export { SequencerUI };