package com.urbanairship.push.embedded;

import android.os.Build;
import com.flurry.org.codehaus.jackson.util.MinimalPrettyPrinter;
import com.google.protobuf.ByteString;
import com.google.protobuf.CodedInputStream;
import com.urbanairship.Logger;
import com.urbanairship.UAirship;
import com.urbanairship.push.PushManager;
import com.urbanairship.push.PushPreferences;
import com.urbanairship.push.embedded.Crypto;
import com.urbanairship.push.proto.Messages;
import com.urbanairship.push.proto.Rpc;
import com.urbanairship.util.Base64;
import com.urbanairship.util.Base64DecoderException;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.net.Socket;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

/* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/2.dex */
public class HeliumClient {
    private BoxOfficeClient boxOfficeClient;
    private CodedInputStream in;
    private OutputStream out;
    private PushPreferences prefs = PushManager.shared().getPreferences();
    private HashMap<String, String> pushIDs = new HashMap<>();

    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/2.dex */
    public class HeliumException extends Exception {
        private static final long serialVersionUID = 1;

        public HeliumException(String str) {
            super(str);
        }
    }

    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/2.dex */
    public class HeliumInvalidPackageException extends HeliumException {
        private static final long serialVersionUID = 1;

        public HeliumInvalidPackageException(String str) {
            super(str);
        }
    }

    public HeliumClient(Socket socket, BoxOfficeClient boxOfficeClient) throws IOException {
        this.boxOfficeClient = boxOfficeClient;
        this.in = CodedInputStream.newInstance(socket.getInputStream());
        this.out = socket.getOutputStream();
    }

    private void handleNotification(Messages.PushNotification pushNotification) {
        String packageName = pushNotification.getPackageName();
        String messageId = pushNotification.getMessageId();
        Logger.verbose("Got app id: " + packageName);
        if (messageAlreadySent(packageName, messageId)) {
            Logger.info("Message " + messageId + " already sent. Discarding.");
            return;
        }
        Logger.info("Message " + messageId + " received.");
        recordMessage(packageName, messageId);
        if (this.prefs.isPushEnabled()) {
            EmbeddedPushManager.deliverPush(pushNotification);
        } else {
            Logger.warn(String.format("Got push notification, but Push is disabled", new Object[0]));
        }
    }

    private void handleRegistration(Messages.RegistrationResponse registrationResponse) throws HeliumInvalidPackageException {
        Iterator<Messages.Relier> it;
        String str;
        Iterator<Messages.Relier> it2;
        String str2;
        Logger.debug("Registration response received for APID: " + this.prefs.getPushId());
        List<Messages.Relier> validReliersList = registrationResponse.getValidReliersList();
        List<Messages.Relier> invalidReliersList = registrationResponse.getInvalidReliersList();
        String str3 = "";
        while (true) {
            str = str3;
            if (!validReliersList.iterator().hasNext()) {
                break;
            }
            str3 = str + it.next().getPackage() + MinimalPrettyPrinter.DEFAULT_ROOT_VALUE_SEPARATOR;
        }
        String str4 = "";
        while (true) {
            str2 = str4;
            if (!invalidReliersList.iterator().hasNext()) {
                break;
            }
            str4 = str2 + it2.next().getPackage() + MinimalPrettyPrinter.DEFAULT_ROOT_VALUE_SEPARATOR;
        }
        Logger.verbose("Valid: " + str);
        Logger.verbose("Invalid: " + str2);
        if (validReliersList.size() > 0) {
            EmbeddedPushManager.sendRegistrationResponse(true);
        } else if (invalidReliersList.size() > 0) {
            throw new HeliumInvalidPackageException("Package name on server does not match the application package name.");
        }
        this.boxOfficeClient.resetFailureCount();
    }

    private boolean messageAlreadySent(String str, String str2) {
        return this.pushIDs.containsKey(str) && this.pushIDs.get(str).equals(str2);
    }

    private static ArrayList<Messages.Relier> protoReliers(String str, String str2) {
        ArrayList<Messages.Relier> arrayList = new ArrayList<>();
        arrayList.add(Messages.Relier.newBuilder().setPackage(str).setAppKey(str2).build());
        return arrayList;
    }

    private void recordMessage(String str, String str2) {
        this.pushIDs.put(str, str2);
    }

    private void sendRequest(Messages.AirshipMethod airshipMethod, ByteString byteString) throws IOException {
        Rpc.Request build = Rpc.Request.newBuilder().setBody(byteString).setMethodId(airshipMethod).build();
        short serializedSize = (short) build.getSerializedSize();
        Logger.verbose("HeliumClient.sendRequest() - Protobuf Request Size: " + ((int) serializedSize));
        ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream(serializedSize + 2);
        byteArrayOutputStream.write((serializedSize >> 8) & 255);
        byteArrayOutputStream.write(serializedSize & 255);
        build.writeTo(byteArrayOutputStream);
        this.out.write(byteArrayOutputStream.toByteArray());
        this.out.flush();
    }

    /* JADX INFO: Access modifiers changed from: protected */
    public void readResponse() throws IOException, HeliumException, Crypto.NullUUID, Base64DecoderException {
        short s = ByteBuffer.wrap(this.in.readRawBytes(2)).getShort();
        Logger.verbose("HeliumClient.readResponse() - Protobuf Response Size: " + ((int) s));
        if (s == 0) {
            Logger.verbose("HeliumClient.readResponse() - Keepalive Read");
        } else if (this.boxOfficeClient.getEncryptionKey() == null) {
            throw new HeliumException("HeliumClient.readResponse() - No Key");
        } else {
            Rpc.Request parseFrom = Rpc.Request.parseFrom(new Crypto(this.boxOfficeClient.getEncryptionKey()).decrypt(Base64.decode(this.in.readRawBytes(s))));
            Logger.verbose("response read");
            if (parseFrom.getMethodId() == Messages.AirshipMethod.REGISTER) {
                handleRegistration(Messages.RegistrationResponse.parseFrom(parseFrom.getBody()));
            } else if (parseFrom.getMethodId() == Messages.AirshipMethod.PUSH_NOTIFICATION) {
                handleNotification(Messages.PushNotification.parseFrom(parseFrom.getBody()));
            } else {
                Logger.error("HeliumClient.readResponse() - Recevied an unknown Helium method type.");
            }
        }
    }

    /* JADX INFO: Access modifiers changed from: protected */
    public void register() throws IOException, Crypto.NullUUID {
        sendRequest(Messages.AirshipMethod.REGISTER, Messages.RegistrationEnvelope.newBuilder().setApid(this.prefs.getPushId()).setRegistration(ByteString.copyFrom(Base64.encodeBytesToBytes(new Crypto(this.boxOfficeClient.getEncryptionKey()).encrypt(Messages.Register.newBuilder().setApid(this.prefs.getPushId()).setOs(Messages.OS.ANDROID).setOsVersion(Build.VERSION.RELEASE).setUaVersion(EmbeddedPushManager.version).setSecret(this.prefs.getPushSecret()).addAllReliers(protoReliers(UAirship.getPackageName(), UAirship.shared().getAirshipConfigOptions().getAppKey())).build().toByteArray())))).build().toByteString());
    }
}
