package com.urbanairship.push;

import android.app.Service;
import android.content.Context;
import android.content.Intent;
import android.net.wifi.WifiManager;
import android.os.IBinder;
import android.os.PowerManager;
import android.provider.Settings;
import com.tapjoy.TapjoyConstants;
import com.urbanairship.AirshipConfigOptions;
import com.urbanairship.Logger;
import com.urbanairship.UAirship;
import com.urbanairship.analytics.PushServiceStartedEvent;
import com.urbanairship.push.c2dm.C2DMPushManager;
import com.urbanairship.push.embedded.EmbeddedPushManager;
import com.urbanairship.push.embedded.Network;

/* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/2.dex */
public class PushService extends Service {
    public static final String ACTION_HEARTBEAT = "com.urbanairship.push.HEARTBEAT";
    public static final String ACTION_START = "com.urbanairship.push.START";
    public static final String ACTION_STOP = "com.urbanairship.push.STOP";
    public boolean started = false;

    private PushPreferences prefs() {
        return PushManager.shared().getPreferences();
    }

    private synchronized void setupService() {
        if (!this.started) {
            wakeWifi();
            if (prefs().getPushId() != null) {
                UAirship.shared().getAnalytics().addEvent(new PushServiceStartedEvent());
            }
            if (prefs().isPushEnabled()) {
                this.started = true;
                Context applicationContext = UAirship.shared().getApplicationContext();
                AirshipConfigOptions airshipConfigOptions = UAirship.shared().getAirshipConfigOptions();
                String appKey = airshipConfigOptions.getAppKey();
                PushManager.shared().getPreferences().setApidUpdateNeeded(true);
                AirshipConfigOptions.TransportType transport = airshipConfigOptions.getTransport();
                if (transport == AirshipConfigOptions.TransportType.HELIUM) {
                    Logger.debug("Starting Helium");
                    if (prefs().getC2DMId() != null) {
                        PushManager.shared().setC2DMId(null);
                        C2DMPushManager.unregister();
                    }
                    EmbeddedPushManager.init(applicationContext, appKey);
                    Logger.debug("Extra APID registrations disabled: " + EmbeddedPushManager.shared().getExtraApidRegistrationsDisabled());
                } else if (transport == AirshipConfigOptions.TransportType.C2DM) {
                    Logger.debug("Starting C2DM");
                    C2DMPushManager.init();
                } else {
                    Logger.debug("Starting Hybrid w/C2DM");
                    C2DMPushManager.init();
                }
            } else {
                Logger.verbose("Push is disabled.  Not starting Push Service.");
                stopSelf();
            }
        }
    }

    private void teardownService() {
        AirshipConfigOptions.TransportType transport = UAirship.shared().getAirshipConfigOptions().getTransport();
        if (transport == AirshipConfigOptions.TransportType.HELIUM || transport == AirshipConfigOptions.TransportType.HYBRID) {
            EmbeddedPushManager.shared().teardown();
        }
        this.started = false;
    }

    private void wakeWifi() {
        boolean z;
        int i = 0;
        if (prefs().isWifiWakeEnabled()) {
            if (-1 == UAirship.getPackageManager().checkPermission("android.permission.ACCESS_WIFI_STATE", UAirship.getPackageName())) {
                Logger.error("AndroidManifest.xml missing required wifi wake permission: android.permission.ACCESS_WIFI_STATE");
                z = false;
            } else {
                z = true;
            }
            if (-1 == UAirship.getPackageManager().checkPermission("android.permission.WAKE_LOCK", UAirship.getPackageName())) {
                Logger.error("AndroidManifest.xml missing required wifi wake permission: android.permission.WAKE_LOCK");
                z = false;
            }
            if (z) {
                final PowerManager.WakeLock newWakeLock = ((PowerManager) UAirship.shared().getApplicationContext().getSystemService("power")).newWakeLock(1, "UALIB");
                WifiManager wifiManager = (WifiManager) UAirship.shared().getApplicationContext().getSystemService(TapjoyConstants.TJC_CONNECTION_TYPE_WIFI);
                final WifiManager.WifiLock createWifiLock = wifiManager.createWifiLock(1, "UALIB");
                Logger.verbose("Network is connected: " + Network.isConnected());
                Logger.verbose("Wifi is enabled " + wifiManager.isWifiEnabled());
                try {
                    i = Settings.System.getInt(getContentResolver(), "wifi_sleep_policy");
                    Logger.verbose("Wifi sleep policy: " + i);
                } catch (Settings.SettingNotFoundException e) {
                    Logger.verbose("Cannot read Wifi sleep policy; assuming WIFI_SLEEP_POLICY_DEFAULT");
                }
                if (Network.isConnected() || !wifiManager.isWifiEnabled() || i == 2) {
                    return;
                }
                Logger.info("No network, but Wifi is enabled. Waking from sleep.");
                newWakeLock.acquire();
                createWifiLock.acquire();
                new Thread() { // from class: com.urbanairship.push.PushService.1
                    @Override // java.lang.Thread, java.lang.Runnable
                    public void run() {
                        try {
                            Thread.sleep(30000L);
                            Logger.verbose("Releasing wake and wifi locks.");
                            createWifiLock.release();
                            if (newWakeLock.isHeld()) {
                                newWakeLock.release();
                            }
                        } catch (InterruptedException e2) {
                            Logger.verbose("Releasing wake and wifi locks.");
                            createWifiLock.release();
                            if (newWakeLock.isHeld()) {
                                newWakeLock.release();
                            }
                        } catch (Throwable th) {
                            Logger.verbose("Releasing wake and wifi locks.");
                            createWifiLock.release();
                            if (newWakeLock.isHeld()) {
                                newWakeLock.release();
                            }
                            throw th;
                        }
                    }
                }.start();
            }
        }
    }

    @Override // android.app.Service
    public IBinder onBind(Intent intent) {
        return null;
    }

    @Override // android.app.Service
    public void onDestroy() {
        Logger.verbose("Push Service destroyed");
        teardownService();
    }

    @Override // android.app.Service
    public void onStart(Intent intent, int i) {
        Logger.debug("Push Service started with intent=" + intent);
        if (!UAirship.shared().isFlying()) {
            Logger.error("PushService started prior to a UAirship.takeOff() call.");
            throw new IllegalStateException("UAirship.takeOff() must be called every time Application.onCreate() is invoked.");
        }
        super.onStart(intent, i);
        if (intent == null || intent.getAction() == null) {
            Logger.warn("Attempted to start service with null intent or action.");
            return;
        }
        String action = intent.getAction();
        if (action.equals(ACTION_STOP)) {
            if (this.started) {
                stopSelf();
            }
        } else if (action.equals(ACTION_START)) {
            setupService();
        } else if (!action.equals(ACTION_HEARTBEAT)) {
            Logger.warn("Unknown action: " + intent.getAction());
        } else {
            Logger.debug("** Heartbeat **");
            if (!this.started) {
                setupService();
                return;
            }
            wakeWifi();
            EmbeddedPushManager.shared().resetStuckConnection();
        }
    }
}
