package com.urbanairship.location;

import android.content.BroadcastReceiver;
import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.ServiceConnection;
import android.location.Criteria;
import android.location.Location;
import android.os.IBinder;
import android.os.RemoteException;
import com.urbanairship.Logger;
import com.urbanairship.UAirship;
import com.urbanairship.analytics.Analytics;
import com.urbanairship.location.ILocationService;
import com.urbanairship.util.ServiceNotBoundException;

/* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/2.dex */
public class UALocationManager {
    public static final String ACTION_LOCATION_SERVICE_BOUND = "com.urbanairship.location.LOCATION_SERVICE_BOUND";
    public static final String ACTION_LOCATION_SERVICE_UNBOUND = "com.urbanairship.location.LOCATION_SERVICE_UNBOUND";
    public static final String ACTION_LOCATION_UPDATE = "com.urbanairship.location.LOCATION_UPDATE";
    public static final String LOCATION_KEY = "com.urbanairship.location.LOCATION";
    private static Context context;
    private static ILocationService locationService;
    private BroadcastReceiver appStateChangeReceiver = new BroadcastReceiver() { // from class: com.urbanairship.location.UALocationManager.2
        @Override // android.content.BroadcastReceiver
        public void onReceive(Context context2, Intent intent) {
            boolean isLocationEnabled = UALocationManager.this.preferences.isLocationEnabled();
            boolean isBackgroundLocationEnabled = UALocationManager.this.preferences.isBackgroundLocationEnabled();
            if (!isLocationEnabled || isBackgroundLocationEnabled) {
                return;
            }
            if (Analytics.ACTION_APP_BACKGROUND.equals(intent.getAction())) {
                UALocationManager.stopAndUnbindService();
            } else if (Analytics.ACTION_APP_FOREGROUND.equals(intent.getAction())) {
                UALocationManager.startAndBindService();
            }
        }
    };
    private LocationPreferences preferences;
    private static final UALocationManager instance = new UALocationManager();
    private static boolean bound = false;
    private static ServiceConnection locationConnection = new ServiceConnection() { // from class: com.urbanairship.location.UALocationManager.1
        @Override // android.content.ServiceConnection
        public void onServiceConnected(ComponentName componentName, IBinder iBinder) {
            ILocationService unused = UALocationManager.locationService = ILocationService.Stub.asInterface(iBinder);
            boolean unused2 = UALocationManager.bound = true;
            UALocationManager.context.sendBroadcast(new Intent(UALocationManager.ACTION_LOCATION_SERVICE_BOUND));
        }

        @Override // android.content.ServiceConnection
        public void onServiceDisconnected(ComponentName componentName) {
            ILocationService unused = UALocationManager.locationService = null;
            boolean unused2 = UALocationManager.bound = false;
            UALocationManager.context.sendBroadcast(new Intent(UALocationManager.ACTION_LOCATION_SERVICE_UNBOUND));
            Logger.info("LocationService unbound.");
        }
    };

    private UALocationManager() {
        context = UAirship.shared().getApplicationContext();
        this.preferences = new LocationPreferences();
        IntentFilter intentFilter = new IntentFilter();
        intentFilter.addAction(Analytics.ACTION_APP_BACKGROUND);
        intentFilter.addAction(Analytics.ACTION_APP_FOREGROUND);
        UAirship.shared().getApplicationContext().registerReceiver(this.appStateChangeReceiver, intentFilter);
    }

    public static void bindService() {
        if (bound) {
            return;
        }
        bound = true;
        context.bindService(new Intent(context, LocationService.class), locationConnection, 1);
    }

    public static void disableBackgroundLocation() {
        Logger.info("Disabling background location");
        instance.preferences.setBackgroundLocationEnabled(false);
        if (UAirship.shared().getAnalytics().isAppInForeground() || !instance.preferences.isLocationEnabled()) {
            return;
        }
        stopAndUnbindService();
    }

    public static void disableLocation() {
        Logger.info("Disable location.");
        instance.preferences.setLocationEnabled(false);
        stopAndUnbindService();
    }

    public static void enableBackgroundLocation() {
        Logger.info("Enabling background location");
        instance.preferences.setBackgroundLocationEnabled(true);
        if (UAirship.shared().getAnalytics().isAppInForeground() || !instance.preferences.isLocationEnabled()) {
            return;
        }
        startAndBindService();
    }

    public static void enableLocation() {
        Logger.info("Enable location.");
        instance.preferences.setLocationEnabled(true);
        if (instance.preferences.isBackgroundLocationEnabled() || UAirship.shared().getAnalytics().isAppInForeground()) {
            startAndBindService();
        }
    }

    private void exceptIfNotBound() {
        if (!bound) {
            throw new ServiceNotBoundException("You must call bindService or enableLocation and wait for the LOCATION_SERVICE_BOUND broadcast before using this method.");
        }
    }

    public static void init() {
        if (!UAirship.shared().isFlying()) {
            throw new IllegalStateException("UAirship.takeOff must be called before UALocationManager.init!");
        }
        if (instance.preferences.isLocationEnabled() && instance.preferences.isBackgroundLocationEnabled()) {
            startService();
        } else {
            Logger.debug("Location or background location are not enabled - deferring service start until foreground event.");
        }
    }

    public static boolean isServiceBound() {
        return bound;
    }

    public static UALocationManager shared() {
        return instance;
    }

    /* JADX INFO: Access modifiers changed from: private */
    public static void startAndBindService() {
        startService();
        bindService();
    }

    private static void startService() {
        Intent intent = new Intent(context, LocationService.class);
        intent.setAction(LocationService.ACTION_START);
        context.startService(intent);
    }

    /* JADX INFO: Access modifiers changed from: private */
    public static void stopAndUnbindService() {
        unbindService();
        stopService();
    }

    private static void stopService() {
        Intent intent = new Intent(context, LocationService.class);
        intent.setAction(LocationService.ACTION_STOP);
        context.stopService(intent);
    }

    public static void unbindService() {
        if (bound) {
            context.unbindService(locationConnection);
        }
        context.sendBroadcast(new Intent(ACTION_LOCATION_SERVICE_UNBOUND));
        bound = false;
    }

    public String getBestProvider() throws RemoteException, ServiceNotBoundException {
        exceptIfNotBound();
        return locationService.getBestProvider();
    }

    public Criteria getCriteria() throws RemoteException, ServiceNotBoundException {
        exceptIfNotBound();
        return locationService.getCriteria();
    }

    public String getCurrentProvider() throws RemoteException, ServiceNotBoundException {
        exceptIfNotBound();
        return locationService.getCurrentProvider();
    }

    public Location getLocation() throws RemoteException, ServiceNotBoundException {
        exceptIfNotBound();
        return locationService.getLocation();
    }

    public LocationPreferences getPreferences() {
        return this.preferences;
    }

    public void recordCurrentLocation() throws RemoteException, ServiceNotBoundException {
        exceptIfNotBound();
        locationService.requestSingleLocationUpdate(null);
    }

    public void recordCurrentLocation(Criteria criteria) throws RemoteException, ServiceNotBoundException {
        exceptIfNotBound();
        locationService.requestSingleLocationUpdate(criteria);
    }

    public void recordLocation(Location location) throws RemoteException, ServiceNotBoundException {
        exceptIfNotBound();
        recordLocation(location, -1, -1);
    }

    public void recordLocation(Location location, int i, int i2) throws RemoteException, ServiceNotBoundException {
        exceptIfNotBound();
        locationService.createLocationEvent(location, i, i2);
    }

    public void resetProviders(Criteria criteria) throws RemoteException, ServiceNotBoundException {
        exceptIfNotBound();
        locationService.setCriteria(criteria);
        locationService.resetProviders();
    }
}
