package com.urbanairship.iap;

import android.app.Activity;
import android.content.ComponentName;
import android.content.Intent;
import android.content.SharedPreferences;
import android.content.pm.PackageManager;
import android.os.Handler;
import com.urbanairship.Logger;
import com.urbanairship.UAirship;
import com.urbanairship.iap.marketinterface.BillingReceiver;
import com.urbanairship.iap.marketinterface.BillingService;
import com.urbanairship.iap.marketinterface.Consts;
import com.urbanairship.iap.marketinterface.ResponseHandler;

/* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/2.dex */
public final class IAPManager {
    private static final String IAP_FIRST_RUN_PREFERENCES = "com.urbanairship.iap.first_run";
    private static final String INITIALIZED_KEY = "initialized";
    private static final IAPManager instance = new IAPManager();
    private int appIcon;
    private BillingService billingService;
    private DownloadManager downloadManager;
    private String downloadPath;
    private IAPEventListener eventListener;
    private Inventory inventory;
    private MarketListener marketListener;
    private PurchaseNotificationBuilder notificationBuilder;
    private String tempPath;
    private String appName = null;
    private boolean isBillingSupported = false;

    private IAPManager() {
    }

    private static void checkRequiredPermission(String str) {
        if (-1 == UAirship.getPackageManager().checkPermission(str, UAirship.getPackageName())) {
            Logger.error("AndroidManifest.xml missing required IAP permission: " + str);
        }
    }

    /* JADX INFO: Access modifiers changed from: package-private */
    public static void firstRun() {
        try {
            SharedPreferences.Editor edit = UAirship.shared().getApplicationContext().getSharedPreferences(IAP_FIRST_RUN_PREFERENCES, 0).edit();
            edit.putBoolean(INITIALIZED_KEY, true);
            edit.commit();
        } catch (Exception e) {
            Logger.error("Error writing to shared preferences");
        }
    }

    public static void init() {
        instance.inventory = new Inventory();
        instance.downloadManager = new DownloadManager();
        instance.notificationBuilder = new BasicPurchaseNotificationBuilder();
        instance.billingService = new BillingService();
        instance.billingService.setContext(UAirship.shared().getApplicationContext());
        instance.appName = UAirship.getAppName();
        instance.appIcon = UAirship.getAppIcon();
        instance.billingService.checkBillingSupported();
        instance.eventListener = new DefaultIAPEventListener();
        if (instance.marketListener != null) {
            ResponseHandler.unregister(instance.marketListener);
        }
        instance.marketListener = new MarketListener(new Handler());
        ResponseHandler.register(instance.marketListener);
    }

    public static boolean isBillingSupported() {
        return instance.isBillingSupported;
    }

    /* JADX INFO: Access modifiers changed from: package-private */
    public static boolean isFirstRun() {
        return !UAirship.shared().getApplicationContext().getSharedPreferences(IAP_FIRST_RUN_PREFERENCES, 0).getBoolean(INITIALIZED_KEY, false);
    }

    /* JADX INFO: Access modifiers changed from: package-private */
    public static void onBillingSupported(boolean z) {
        instance.isBillingSupported = z;
        if (instance.eventListener != null) {
            instance.eventListener.billingSupported(z);
        }
    }

    public static IAPManager shared() {
        return instance;
    }

    public static void tearDown() {
        instance.billingService.unbind();
        ResponseHandler.unregister(instance.marketListener);
    }

    public static void validateManifest() {
        PackageManager packageManager = UAirship.getPackageManager();
        try {
            packageManager.getReceiverInfo(new ComponentName(UAirship.getPackageName(), BillingReceiver.class.getCanonicalName()), 128);
            Intent intent = new Intent(Consts.ACTION_NOTIFY);
            if (packageManager.queryBroadcastReceivers(intent, 0).isEmpty()) {
                Logger.error("AndroidManifest.xml's " + BillingReceiver.class.getCanonicalName() + " declaration missing required " + intent.getAction() + " filter");
            }
            Intent intent2 = new Intent(Consts.ACTION_RESPONSE_CODE);
            if (packageManager.queryBroadcastReceivers(intent2, 0).isEmpty()) {
                Logger.error("AndroidManifest.xml's " + BillingReceiver.class.getCanonicalName() + " declaration missing required " + intent2.getAction() + " filter");
            }
            Intent intent3 = new Intent(Consts.ACTION_PURCHASE_STATE_CHANGED);
            if (packageManager.queryBroadcastReceivers(intent3, 0).isEmpty()) {
                Logger.error("AndroidManifest.xml's " + BillingReceiver.class.getCanonicalName() + " declaration missing required " + intent3.getAction() + " filter");
            }
        } catch (PackageManager.NameNotFoundException e) {
            Logger.error("AndroidManifest.xml missing required receiver: " + BillingReceiver.class.getCanonicalName());
        }
        try {
            packageManager.getServiceInfo(new ComponentName(UAirship.getPackageName(), BillingService.class.getCanonicalName()), 128);
        } catch (PackageManager.NameNotFoundException e2) {
            Logger.error("AndroidManifest.xml missing required service: " + BillingService.class.getCanonicalName());
        }
        checkRequiredPermission("com.android.vending.BILLING");
    }

    public int getAppIcon() {
        return this.appIcon;
    }

    public String getAppName() {
        return this.appName;
    }

    BillingService getBillingService() {
        return this.billingService;
    }

    /* JADX INFO: Access modifiers changed from: package-private */
    public DownloadManager getDownloadManager() {
        return this.downloadManager;
    }

    public String getDownloadPath() {
        return this.downloadPath != null ? this.downloadPath : "iap/downloads/" + UAirship.getAppName() + "/";
    }

    public IAPEventListener getEventListener() {
        return this.eventListener;
    }

    public Inventory getInventory() {
        return this.inventory;
    }

    public PurchaseNotificationBuilder getNotificationBuilder() {
        return this.notificationBuilder;
    }

    public String getTempPath() {
        return this.tempPath != null ? this.tempPath : "iap/temp/" + UAirship.getAppName() + "/";
    }

    /* JADX INFO: Access modifiers changed from: package-private */
    public boolean payForProduct(Activity activity, Product product) {
        if (!this.isBillingSupported) {
            Logger.error("Billing is not supported on this version of Android Market");
            return false;
        } else if (this.billingService.requestPurchase(activity, product.getIdentifier())) {
            return true;
        } else {
            if (this.eventListener != null) {
                this.eventListener.marketUnavailable(product);
                return false;
            }
            return false;
        }
    }

    public void restoreTransactions() {
        IAPEventListener eventListener = getEventListener();
        if (eventListener != null) {
            eventListener.restoreStarted();
        }
        getBillingService().restoreTransactions();
    }

    public void setDownloadPath(String str) {
        this.downloadPath = str;
    }

    public void setEventListener(IAPEventListener iAPEventListener) {
        this.eventListener = iAPEventListener;
    }

    public void setNotificationBuilder(PurchaseNotificationBuilder purchaseNotificationBuilder) {
        this.notificationBuilder = purchaseNotificationBuilder;
    }

    public void setTempPath(String str) {
        this.tempPath = str;
    }
}
