package com.urbanairship.analytics;

import android.os.Build;
import com.tapjoy.TapjoyConstants;
import com.urbanairship.AirshipConfigOptions;
import com.urbanairship.Logger;
import com.urbanairship.Preferences;
import com.urbanairship.UAirship;
import com.urbanairship.push.PushManager;
import com.urbanairship.restclient.Request;
import com.urbanairship.restclient.Response;
import java.util.Collections;
import java.util.Map;
import org.apache.http.Header;

/* JADX INFO: Access modifiers changed from: package-private */
/* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/2.dex */
public class EventUploadManager implements Runnable {
    public static final int DEFAULT_BATCH_SIZE = 102400;
    public static final String LAST_SEND_KEY = "LAST_SEND";
    public static final int MAX_BATCH_SIZE = 512000;
    public static final String MAX_BATCH_SIZE_KEY = "MAX_BATCH_SIZE";
    public static final int MAX_TOTAL_DB_SIZE = 5120000;
    public static final String MAX_TOTAL_DB_SIZE_KEY = "MAX_TOTAL_DB_SIZE";
    public static final String MAX_WAIT_KEY = "MAX_WAIT";
    public static final int MAX_WAIT_MS = 604800000;
    public static final String MIN_BATCH_INTERVAL_KEY = "MIN_BATCH_INTERVAL";
    public static final int MIN_BATCH_INTERVAL_MS = 60000;
    private int batchSize;
    private Thread currentThread;
    private EventDataManager dataManager;
    private long lastSendTime;
    private int maxBatchSize;
    private int maxTotalDBSize;
    private int maxWait;
    private int minBatchInterval;
    private Preferences prefs = new Preferences("com.urbanairship.analytics");
    private volatile boolean running;
    private boolean useGzip;

    public EventUploadManager(EventDataManager eventDataManager) {
        loadPrefs();
        this.batchSize = DEFAULT_BATCH_SIZE;
        this.dataManager = eventDataManager;
        this.useGzip = true;
    }

    private void loadPrefs() {
        this.maxTotalDBSize = this.prefs.getInt(MAX_TOTAL_DB_SIZE_KEY, MAX_TOTAL_DB_SIZE);
        this.maxBatchSize = this.prefs.getInt(MAX_BATCH_SIZE_KEY, MAX_BATCH_SIZE);
        this.maxWait = this.prefs.getInt(MAX_WAIT_KEY, MAX_WAIT_MS);
        this.minBatchInterval = this.prefs.getInt(MIN_BATCH_INTERVAL_KEY, 60000);
        this.lastSendTime = this.prefs.getLong(LAST_SEND_KEY, System.currentTimeMillis());
    }

    private void saveHeaders(Response response) {
        int i = MAX_TOTAL_DB_SIZE;
        Header firstHeader = response.getFirstHeader("X-UA-Max-Total");
        Header firstHeader2 = response.getFirstHeader("X-UA-Max-Batch");
        Header firstHeader3 = response.getFirstHeader("X-UA-Max-Wait");
        Header firstHeader4 = response.getFirstHeader("X-UA-Min-Batch-Interval");
        if (firstHeader != null) {
            int min = Math.min(Integer.parseInt(firstHeader.getValue()) * 1024, (int) MAX_TOTAL_DB_SIZE);
            if (min > 0) {
                i = min;
            }
            this.maxTotalDBSize = i;
        }
        if (firstHeader2 != null) {
            int min2 = Math.min(Integer.parseInt(firstHeader2.getValue()) * 1024, (int) MAX_BATCH_SIZE);
            if (min2 <= 0) {
                min2 = 512000;
            }
            this.maxBatchSize = min2;
        }
        if (firstHeader3 != null) {
            int min3 = Math.min(Integer.parseInt(firstHeader3.getValue()), (int) MAX_WAIT_MS);
            if (min3 <= 0) {
                min3 = 604800000;
            }
            this.maxWait = min3;
        }
        if (firstHeader4 != null) {
            int max = Math.max(Integer.parseInt(firstHeader4.getValue()), 60000);
            if (max <= 0) {
                max = 60000;
            }
            this.minBatchInterval = max;
        }
        savePrefs();
    }

    private void savePrefs() {
        this.prefs.putInt(MAX_TOTAL_DB_SIZE_KEY, this.maxTotalDBSize);
        this.prefs.putInt(MAX_BATCH_SIZE_KEY, this.maxBatchSize);
        this.prefs.putInt(MAX_WAIT_KEY, this.maxWait);
        this.prefs.putInt(MIN_BATCH_INTERVAL_KEY, this.minBatchInterval);
        this.prefs.putLong(LAST_SEND_KEY, this.lastSendTime);
    }

    /* JADX WARN: Removed duplicated region for block: B:22:0x00f2  */
    /* JADX WARN: Removed duplicated region for block: B:25:0x0129  */
    /* JADX WARN: Removed duplicated region for block: B:32:0x015c  */
    /*
        Code decompiled incorrectly, please refer to instructions dump.
        To view partially-correct code enable 'Show inconsistent code' option in preferences
    */
    private boolean send(java.util.Collection<java.lang.String> r9) {
        /*
            Method dump skipped, instructions count: 355
            To view this dump change 'Code comments level' option to 'DEBUG'
        */
        throw new UnsupportedOperationException("Method not decompiled: com.urbanairship.analytics.EventUploadManager.send(java.util.Collection):boolean");
    }

    private void setHeaders(Request request) {
        request.setHeader("X-UA-Device-Family", TapjoyConstants.TJC_DEVICE_PLATFORM_TYPE);
        request.setHeader("X-UA-Sent-At", Double.toString(System.currentTimeMillis() / 1000.0d));
        request.setHeader("X-UA-Package-Name", UAirship.getPackageName());
        request.setHeader("X-UA-Package-Version", UAirship.getPackageInfo().versionName);
        String hashedDeviceId = Analytics.getHashedDeviceId();
        if (hashedDeviceId.length() > 0) {
            request.setHeader("X-UA-Device-ID", hashedDeviceId);
        }
        AirshipConfigOptions airshipConfigOptions = UAirship.shared().getAirshipConfigOptions();
        request.setHeader("X-UA-App-Key", airshipConfigOptions.getAppKey());
        request.setHeader("X-UA-In-Production", Boolean.toString(airshipConfigOptions.inProduction));
        request.setHeader("X-UA-Device-Model", Build.MODEL);
        request.setHeader("X-UA-OS-Version", Build.VERSION.RELEASE);
        request.setHeader("X-UA-Lib-Version", UAirship.getVersion());
        String pushId = PushManager.shared().getPreferences().getPushId();
        if (pushId == null || pushId.length() <= 0) {
            return;
        }
        request.setHeader("X-UA-Apid", pushId);
    }

    public long getLastSendTime() {
        return this.prefs.getLong(LAST_SEND_KEY, System.currentTimeMillis());
    }

    public int getMaxTotalDBSize() {
        return this.maxTotalDBSize;
    }

    @Override // java.lang.Runnable
    public void run() {
        while (this.running) {
            long max = Math.max(0L, (this.lastSendTime + this.minBatchInterval) - System.currentTimeMillis());
            if (max > 0) {
                try {
                    Thread.sleep(max);
                } catch (InterruptedException e) {
                    Logger.info("InterruptedException in EventUploadManager. Bailing!");
                    this.running = false;
                    return;
                }
            }
            if (!this.running) {
                return;
            }
            this.lastSendTime = System.currentTimeMillis();
            this.prefs.putLong(LAST_SEND_KEY, this.lastSendTime);
            int eventCount = this.dataManager.getEventCount();
            if (eventCount <= 0) {
                this.running = false;
                Logger.info("Exiting analytics upload thread.");
                return;
            }
            Map<Long, String> oldestEvents = this.dataManager.getOldestEvents(this.batchSize / (this.dataManager.getDatabaseSize() / eventCount));
            if (send(oldestEvents.values())) {
                this.dataManager.deleteEventsOlderThan(((Long) Collections.max(oldestEvents.keySet())).longValue());
            }
        }
    }

    public void startUploadingIfNecessary() {
        if (this.running) {
            return;
        }
        Logger.verbose("EventUploadManager - starting upload thread");
        this.running = true;
        this.currentThread = new Thread(this);
        this.currentThread.start();
    }

    public void stopUpload() {
        this.running = false;
    }
}
