package com.flurry.org.codehaus.jackson.map.deser.std;

import com.flurry.org.codehaus.jackson.JsonProcessingException;
import com.flurry.org.codehaus.jackson.io.NumberInput;
import com.flurry.org.codehaus.jackson.map.DeserializationContext;
import com.flurry.org.codehaus.jackson.map.JsonMappingException;
import com.flurry.org.codehaus.jackson.map.KeyDeserializer;
import com.flurry.org.codehaus.jackson.map.introspect.AnnotatedMethod;
import com.flurry.org.codehaus.jackson.map.util.ClassUtil;
import com.flurry.org.codehaus.jackson.map.util.EnumResolver;
import java.io.IOException;
import java.lang.reflect.Constructor;
import java.lang.reflect.Method;
import java.util.Calendar;
import java.util.Date;
import java.util.UUID;

/* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/1.dex */
public abstract class StdKeyDeserializer extends KeyDeserializer {
    protected final Class<?> _keyClass;

    protected abstract Object _parse(String str, DeserializationContext deserializationContext) throws Exception;

    /* JADX INFO: Access modifiers changed from: protected */
    public StdKeyDeserializer(Class<?> cls) {
        this._keyClass = cls;
    }

    @Override // com.flurry.org.codehaus.jackson.map.KeyDeserializer
    public final Object deserializeKey(String key, DeserializationContext ctxt) throws IOException, JsonProcessingException {
        if (key == null) {
            return null;
        }
        try {
            Object result = _parse(key, ctxt);
            if (result != null) {
                return result;
            }
            throw ctxt.weirdKeyException(this._keyClass, key, "not a valid representation");
        } catch (Exception re) {
            throw ctxt.weirdKeyException(this._keyClass, key, "not a valid representation: " + re.getMessage());
        }
    }

    public Class<?> getKeyClass() {
        return this._keyClass;
    }

    protected int _parseInt(String key) throws IllegalArgumentException {
        return Integer.parseInt(key);
    }

    protected long _parseLong(String key) throws IllegalArgumentException {
        return Long.parseLong(key);
    }

    protected double _parseDouble(String key) throws IllegalArgumentException {
        return NumberInput.parseDouble(key);
    }

    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/1.dex */
    static final class StringKD extends StdKeyDeserializer {
        private static final StringKD sString = new StringKD(String.class);
        private static final StringKD sObject = new StringKD(Object.class);

        private StringKD(Class<?> nominalType) {
            super(nominalType);
        }

        public static StringKD forType(Class<?> nominalType) {
            if (nominalType == String.class) {
                return sString;
            }
            if (nominalType == Object.class) {
                return sObject;
            }
            return new StringKD(nominalType);
        }

        @Override // com.flurry.org.codehaus.jackson.map.deser.std.StdKeyDeserializer
        public String _parse(String key, DeserializationContext ctxt) throws JsonMappingException {
            return key;
        }
    }

    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/1.dex */
    static final class BoolKD extends StdKeyDeserializer {
        /* JADX INFO: Access modifiers changed from: package-private */
        public BoolKD() {
            super(Boolean.class);
        }

        @Override // com.flurry.org.codehaus.jackson.map.deser.std.StdKeyDeserializer
        public Boolean _parse(String key, DeserializationContext ctxt) throws JsonMappingException {
            if ("true".equals(key)) {
                return Boolean.TRUE;
            }
            if ("false".equals(key)) {
                return Boolean.FALSE;
            }
            throw ctxt.weirdKeyException(this._keyClass, key, "value not 'true' or 'false'");
        }
    }

    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/1.dex */
    static final class ByteKD extends StdKeyDeserializer {
        /* JADX INFO: Access modifiers changed from: package-private */
        public ByteKD() {
            super(Byte.class);
        }

        @Override // com.flurry.org.codehaus.jackson.map.deser.std.StdKeyDeserializer
        public Byte _parse(String key, DeserializationContext ctxt) throws JsonMappingException {
            int value = _parseInt(key);
            if (value < -128 || value > 127) {
                throw ctxt.weirdKeyException(this._keyClass, key, "overflow, value can not be represented as 8-bit value");
            }
            return Byte.valueOf((byte) value);
        }
    }

    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/1.dex */
    static final class ShortKD extends StdKeyDeserializer {
        /* JADX INFO: Access modifiers changed from: package-private */
        public ShortKD() {
            super(Integer.class);
        }

        @Override // com.flurry.org.codehaus.jackson.map.deser.std.StdKeyDeserializer
        public Short _parse(String key, DeserializationContext ctxt) throws JsonMappingException {
            int value = _parseInt(key);
            if (value < -32768 || value > 32767) {
                throw ctxt.weirdKeyException(this._keyClass, key, "overflow, value can not be represented as 16-bit value");
            }
            return Short.valueOf((short) value);
        }
    }

    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/1.dex */
    static final class CharKD extends StdKeyDeserializer {
        /* JADX INFO: Access modifiers changed from: package-private */
        public CharKD() {
            super(Character.class);
        }

        @Override // com.flurry.org.codehaus.jackson.map.deser.std.StdKeyDeserializer
        public Character _parse(String key, DeserializationContext ctxt) throws JsonMappingException {
            if (key.length() == 1) {
                return Character.valueOf(key.charAt(0));
            }
            throw ctxt.weirdKeyException(this._keyClass, key, "can only convert 1-character Strings");
        }
    }

    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/1.dex */
    static final class IntKD extends StdKeyDeserializer {
        /* JADX INFO: Access modifiers changed from: package-private */
        public IntKD() {
            super(Integer.class);
        }

        @Override // com.flurry.org.codehaus.jackson.map.deser.std.StdKeyDeserializer
        public Integer _parse(String key, DeserializationContext ctxt) throws JsonMappingException {
            return Integer.valueOf(_parseInt(key));
        }
    }

    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/1.dex */
    static final class LongKD extends StdKeyDeserializer {
        /* JADX INFO: Access modifiers changed from: package-private */
        public LongKD() {
            super(Long.class);
        }

        @Override // com.flurry.org.codehaus.jackson.map.deser.std.StdKeyDeserializer
        public Long _parse(String key, DeserializationContext ctxt) throws JsonMappingException {
            return Long.valueOf(_parseLong(key));
        }
    }

    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/1.dex */
    static final class DoubleKD extends StdKeyDeserializer {
        /* JADX INFO: Access modifiers changed from: package-private */
        public DoubleKD() {
            super(Double.class);
        }

        @Override // com.flurry.org.codehaus.jackson.map.deser.std.StdKeyDeserializer
        public Double _parse(String key, DeserializationContext ctxt) throws JsonMappingException {
            return Double.valueOf(_parseDouble(key));
        }
    }

    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/1.dex */
    static final class FloatKD extends StdKeyDeserializer {
        /* JADX INFO: Access modifiers changed from: package-private */
        public FloatKD() {
            super(Float.class);
        }

        @Override // com.flurry.org.codehaus.jackson.map.deser.std.StdKeyDeserializer
        public Float _parse(String key, DeserializationContext ctxt) throws JsonMappingException {
            return Float.valueOf((float) _parseDouble(key));
        }
    }

    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/1.dex */
    static final class EnumKD extends StdKeyDeserializer {
        protected final AnnotatedMethod _factory;
        protected final EnumResolver<?> _resolver;

        /* JADX INFO: Access modifiers changed from: protected */
        public EnumKD(EnumResolver<?> er, AnnotatedMethod factory) {
            super(er.getEnumClass());
            this._resolver = er;
            this._factory = factory;
        }

        @Override // com.flurry.org.codehaus.jackson.map.deser.std.StdKeyDeserializer
        public Object _parse(String key, DeserializationContext ctxt) throws JsonMappingException {
            if (this._factory != null) {
                try {
                    return this._factory.call1(key);
                } catch (Exception e) {
                    ClassUtil.unwrapAndThrowAsIAE(e);
                }
            }
            Object findEnum = this._resolver.findEnum(key);
            if (findEnum == null) {
                throw ctxt.weirdKeyException(this._keyClass, key, "not one of values for Enum class");
            }
            return findEnum;
        }
    }

    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/1.dex */
    static final class StringCtorKeyDeserializer extends StdKeyDeserializer {
        protected final Constructor<?> _ctor;

        public StringCtorKeyDeserializer(Constructor<?> ctor) {
            super(ctor.getDeclaringClass());
            this._ctor = ctor;
        }

        @Override // com.flurry.org.codehaus.jackson.map.deser.std.StdKeyDeserializer
        public Object _parse(String key, DeserializationContext ctxt) throws Exception {
            return this._ctor.newInstance(key);
        }
    }

    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/1.dex */
    static final class StringFactoryKeyDeserializer extends StdKeyDeserializer {
        final Method _factoryMethod;

        public StringFactoryKeyDeserializer(Method fm) {
            super(fm.getDeclaringClass());
            this._factoryMethod = fm;
        }

        @Override // com.flurry.org.codehaus.jackson.map.deser.std.StdKeyDeserializer
        public Object _parse(String key, DeserializationContext ctxt) throws Exception {
            return this._factoryMethod.invoke(null, key);
        }
    }

    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/1.dex */
    static final class DateKD extends StdKeyDeserializer {
        /* JADX INFO: Access modifiers changed from: protected */
        public DateKD() {
            super(Date.class);
        }

        @Override // com.flurry.org.codehaus.jackson.map.deser.std.StdKeyDeserializer
        public Date _parse(String key, DeserializationContext ctxt) throws IllegalArgumentException, JsonMappingException {
            return ctxt.parseDate(key);
        }
    }

    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/1.dex */
    static final class CalendarKD extends StdKeyDeserializer {
        /* JADX INFO: Access modifiers changed from: protected */
        public CalendarKD() {
            super(Calendar.class);
        }

        @Override // com.flurry.org.codehaus.jackson.map.deser.std.StdKeyDeserializer
        public Calendar _parse(String key, DeserializationContext ctxt) throws IllegalArgumentException, JsonMappingException {
            Date date = ctxt.parseDate(key);
            if (date == null) {
                return null;
            }
            return ctxt.constructCalendar(date);
        }
    }

    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/1.dex */
    static final class UuidKD extends StdKeyDeserializer {
        /* JADX INFO: Access modifiers changed from: protected */
        public UuidKD() {
            super(UUID.class);
        }

        @Override // com.flurry.org.codehaus.jackson.map.deser.std.StdKeyDeserializer
        public UUID _parse(String key, DeserializationContext ctxt) throws IllegalArgumentException, JsonMappingException {
            return UUID.fromString(key);
        }
    }
}
