package com.flurry.org.apache.avro.specific;

import com.flurry.org.apache.avro.AvroRuntimeException;
import com.flurry.org.apache.avro.AvroTypeException;
import com.flurry.org.apache.avro.Protocol;
import com.flurry.org.apache.avro.Schema;
import com.flurry.org.apache.avro.generic.GenericData;
import com.flurry.org.apache.avro.io.DatumReader;
import com.prime31.util.IabHelper;
import com.tapjoy.TapjoyVideoObject;
import java.lang.reflect.Constructor;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.nio.ByteBuffer;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.WeakHashMap;
import java.util.concurrent.ConcurrentHashMap;

/* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
public class SpecificData extends GenericData {
    private Map<String, Class> classCache;
    private final ClassLoader classLoader;
    private final WeakHashMap<Type, Schema> schemaCache;
    private static final SpecificData INSTANCE = new SpecificData();
    private static final Class<?>[] NO_ARG = new Class[0];
    private static final Class<?>[] SCHEMA_ARG = {Schema.class};
    private static final Map<Class, Constructor> CTOR_CACHE = new ConcurrentHashMap();
    private static final Class NO_CLASS = new Object() { // from class: com.flurry.org.apache.avro.specific.SpecificData.1
    }.getClass();
    private static final Schema NULL_SCHEMA = Schema.create(Schema.Type.NULL);

    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public interface SchemaConstructable {
    }

    /* JADX INFO: Access modifiers changed from: protected */
    public SpecificData() {
        this(SpecificData.class.getClassLoader());
    }

    public SpecificData(ClassLoader classLoader) {
        this.classCache = new ConcurrentHashMap();
        this.schemaCache = new WeakHashMap<>();
        this.classLoader = classLoader;
    }

    @Override // com.flurry.org.apache.avro.generic.GenericData
    public DatumReader createDatumReader(Schema schema) {
        return new SpecificDatumReader(schema, schema, this);
    }

    public static SpecificData get() {
        return INSTANCE;
    }

    /* JADX INFO: Access modifiers changed from: protected */
    @Override // com.flurry.org.apache.avro.generic.GenericData
    public boolean isEnum(Object datum) {
        return (datum instanceof Enum) || super.isEnum(datum);
    }

    /* JADX INFO: Access modifiers changed from: protected */
    @Override // com.flurry.org.apache.avro.generic.GenericData
    public Schema getEnumSchema(Object datum) {
        return datum instanceof Enum ? getSchema(datum.getClass()) : super.getEnumSchema(datum);
    }

    public Class getClass(Schema schema) {
        switch (AnonymousClass2.$SwitchMap$org$apache$avro$Schema$Type[schema.getType().ordinal()]) {
            case 1:
            case 2:
            case 3:
                String name = schema.getFullName();
                if (name == null) {
                    return null;
                }
                Class c = this.classCache.get(name);
                if (c == null) {
                    try {
                        c = this.classLoader.loadClass(getClassName(schema));
                    } catch (ClassNotFoundException e) {
                        c = NO_CLASS;
                    }
                    this.classCache.put(name, c);
                }
                if (c == NO_CLASS) {
                    c = null;
                }
                return c;
            case 4:
                return List.class;
            case 5:
                return Map.class;
            case 6:
                List<Schema> types = schema.getTypes();
                if (types.size() == 2 && types.contains(NULL_SCHEMA)) {
                    return getClass(types.get(types.get(0).equals(NULL_SCHEMA) ? 1 : 0));
                }
                return Object.class;
            case IabHelper.BILLING_RESPONSE_RESULT_ITEM_ALREADY_OWNED /* 7 */:
                if ("String".equals(schema.getProp("avro.java.string"))) {
                    return String.class;
                }
                return CharSequence.class;
            case 8:
                return ByteBuffer.class;
            case 9:
                return Integer.TYPE;
            case TapjoyVideoObject.BUTTON_MAX /* 10 */:
                return Long.TYPE;
            case 11:
                return Float.TYPE;
            case 12:
                return Double.TYPE;
            case 13:
                return Boolean.TYPE;
            case 14:
                return Void.TYPE;
            default:
                throw new AvroRuntimeException("Unknown type: " + schema);
        }
    }

    public static String getClassName(Schema schema) {
        String namespace = schema.getNamespace();
        String name = schema.getName();
        if (namespace != null && !"".equals(namespace)) {
            String dot = namespace.endsWith("$") ? "" : ".";
            return namespace + dot + name;
        }
        return name;
    }

    public Schema getSchema(Type type) {
        Schema schema = this.schemaCache.get(type);
        if (schema == null) {
            Schema schema2 = createSchema(type, new LinkedHashMap());
            this.schemaCache.put(type, schema2);
            return schema2;
        }
        return schema;
    }

    /* JADX INFO: Access modifiers changed from: protected */
    public Schema createSchema(Type type, Map<String, Schema> names) {
        if ((type instanceof Class) && CharSequence.class.isAssignableFrom((Class) type)) {
            return Schema.create(Schema.Type.STRING);
        }
        if (type == ByteBuffer.class) {
            return Schema.create(Schema.Type.BYTES);
        }
        if (type == Integer.class || type == Integer.TYPE) {
            return Schema.create(Schema.Type.INT);
        }
        if (type == Long.class || type == Long.TYPE) {
            return Schema.create(Schema.Type.LONG);
        }
        if (type == Float.class || type == Float.TYPE) {
            return Schema.create(Schema.Type.FLOAT);
        }
        if (type == Double.class || type == Double.TYPE) {
            return Schema.create(Schema.Type.DOUBLE);
        }
        if (type == Boolean.class || type == Boolean.TYPE) {
            return Schema.create(Schema.Type.BOOLEAN);
        }
        if (type == Void.class || type == Void.TYPE) {
            return Schema.create(Schema.Type.NULL);
        }
        if (type instanceof ParameterizedType) {
            ParameterizedType ptype = (ParameterizedType) type;
            Class raw = (Class) ptype.getRawType();
            Type[] params = ptype.getActualTypeArguments();
            if (Collection.class.isAssignableFrom(raw)) {
                if (params.length != 1) {
                    throw new AvroTypeException("No array type specified.");
                }
                return Schema.createArray(createSchema(params[0], names));
            } else if (Map.class.isAssignableFrom(raw)) {
                Type key = params[0];
                Type value = params[1];
                if (!(type instanceof Class) || !CharSequence.class.isAssignableFrom((Class) type)) {
                    throw new AvroTypeException("Map key class not CharSequence: " + key);
                }
                return Schema.createMap(createSchema(value, names));
            } else {
                return createSchema(raw, names);
            }
        } else if (type instanceof Class) {
            Class c = (Class) type;
            String fullName = c.getName();
            Schema schema = names.get(fullName);
            if (schema == null) {
                try {
                    schema = (Schema) c.getDeclaredField("SCHEMA$").get(null);
                    if (!fullName.equals(getClassName(schema))) {
                        schema = Schema.parse(schema.toString().replace(schema.getNamespace(), c.getPackage().getName()));
                    }
                } catch (IllegalAccessException e) {
                    throw new AvroRuntimeException(e);
                } catch (NoSuchFieldException e2) {
                    throw new AvroRuntimeException("Not a Specific class: " + c);
                }
            }
            names.put(fullName, schema);
            return schema;
        } else {
            throw new AvroTypeException("Unknown type: " + type);
        }
    }

    public Protocol getProtocol(Class iface) {
        try {
            Protocol p = (Protocol) iface.getDeclaredField("PROTOCOL").get(null);
            return !p.getNamespace().equals(iface.getPackage().getName()) ? Protocol.parse(p.toString().replace(p.getNamespace(), iface.getPackage().getName())) : p;
        } catch (IllegalAccessException e) {
            throw new AvroRuntimeException(e);
        } catch (NoSuchFieldException e2) {
            throw new AvroRuntimeException("Not a Specific protocol: " + iface);
        }
    }

    /* JADX INFO: Access modifiers changed from: protected */
    @Override // com.flurry.org.apache.avro.generic.GenericData
    public int compare(Object o1, Object o2, Schema s, boolean eq) {
        switch (s.getType()) {
            case ENUM:
                if (o1 instanceof Enum) {
                    return ((Enum) o1).ordinal() - ((Enum) o2).ordinal();
                }
                break;
        }
        return super.compare(o1, o2, s, eq);
    }

    public static Object newInstance(Class c, Schema s) {
        boolean useSchema = SchemaConstructable.class.isAssignableFrom(c);
        try {
            Constructor meth = CTOR_CACHE.get(c);
            if (meth == null) {
                meth = c.getDeclaredConstructor(useSchema ? SCHEMA_ARG : NO_ARG);
                meth.setAccessible(true);
                CTOR_CACHE.put(c, meth);
            }
            Object result = meth.newInstance(useSchema ? new Object[]{s} : null);
            return result;
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    @Override // com.flurry.org.apache.avro.generic.GenericData
    public Object createFixed(Object old, Schema schema) {
        Class c = get().getClass(schema);
        return c == null ? super.createFixed(old, schema) : !c.isInstance(old) ? newInstance(c, schema) : old;
    }

    @Override // com.flurry.org.apache.avro.generic.GenericData
    public Object newRecord(Object old, Schema schema) {
        Class c = get().getClass(schema);
        return c == null ? super.newRecord(old, schema) : !c.isInstance(old) ? newInstance(c, schema) : old;
    }
}
