package com.flurry.org.apache.avro.reflect;

import com.flurry.org.apache.avro.AvroRemoteException;
import com.flurry.org.apache.avro.AvroRuntimeException;
import com.flurry.org.apache.avro.AvroTypeException;
import com.flurry.org.apache.avro.Protocol;
import com.flurry.org.apache.avro.Schema;
import com.flurry.org.apache.avro.generic.GenericContainer;
import com.flurry.org.apache.avro.generic.GenericFixed;
import com.flurry.org.apache.avro.generic.IndexedRecord;
import com.flurry.org.apache.avro.io.BinaryData;
import com.flurry.org.apache.avro.io.DatumReader;
import com.flurry.org.apache.avro.specific.FixedSize;
import com.flurry.org.apache.avro.specific.SpecificData;
import com.flurry.org.codehaus.jackson.JsonNode;
import com.flurry.org.codehaus.jackson.node.NullNode;
import com.thoughtworks.paranamer.CachingParanamer;
import com.thoughtworks.paranamer.Paranamer;
import java.lang.annotation.Annotation;
import java.lang.reflect.Array;
import java.lang.reflect.Field;
import java.lang.reflect.GenericArrayType;
import java.lang.reflect.Method;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
public class ReflectData extends SpecificData {
    static final String CLASS_PROP = "java-class";
    static final String ELEMENT_PROP = "java-element-class";
    private final Paranamer paranamer = new CachingParanamer();
    private static final ReflectData INSTANCE = new ReflectData();
    private static final Map<Class, Map<String, Field>> FIELD_CACHE = new ConcurrentHashMap();
    private static final Class BYTES_CLASS = new byte[0].getClass();
    private static final Schema THROWABLE_MESSAGE = makeNullable(Schema.create(Schema.Type.STRING));

    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public static class AllowNull extends ReflectData {
        private static final AllowNull INSTANCE = new AllowNull();

        public static AllowNull get() {
            return INSTANCE;
        }

        @Override // com.flurry.org.apache.avro.reflect.ReflectData
        protected Schema createFieldSchema(Field field, Map<String, Schema> names) {
            Schema schema = super.createFieldSchema(field, names);
            return makeNullable(schema);
        }
    }

    protected ReflectData() {
    }

    public static ReflectData get() {
        return INSTANCE;
    }

    @Override // com.flurry.org.apache.avro.specific.SpecificData, com.flurry.org.apache.avro.generic.GenericData
    public DatumReader createDatumReader(Schema schema) {
        return new ReflectDatumReader(schema, schema, this);
    }

    @Override // com.flurry.org.apache.avro.generic.GenericData
    public void setField(Object record, String name, int position, Object o) {
        if (record instanceof IndexedRecord) {
            super.setField(record, name, position, o);
            return;
        }
        try {
            getField(record.getClass(), name).set(record, o);
        } catch (IllegalAccessException e) {
            throw new AvroRuntimeException(e);
        }
    }

    @Override // com.flurry.org.apache.avro.generic.GenericData
    public Object getField(Object record, String name, int position) {
        if (record instanceof IndexedRecord) {
            return super.getField(record, name, position);
        }
        try {
            return getField(record.getClass(), name).get(record);
        } catch (IllegalAccessException e) {
            throw new AvroRuntimeException(e);
        }
    }

    /* JADX INFO: Access modifiers changed from: protected */
    @Override // com.flurry.org.apache.avro.generic.GenericData
    public boolean isRecord(Object datum) {
        if (datum == null) {
            return false;
        }
        if (super.isRecord(datum)) {
            return true;
        }
        if (datum instanceof Collection) {
            return false;
        }
        return getSchema(datum.getClass()).getType() == Schema.Type.RECORD;
    }

    @Override // com.flurry.org.apache.avro.generic.GenericData
    protected boolean isArray(Object datum) {
        if (datum == null) {
            return false;
        }
        return (datum instanceof Collection) || datum.getClass().isArray();
    }

    /* JADX INFO: Access modifiers changed from: protected */
    @Override // com.flurry.org.apache.avro.generic.GenericData
    public boolean isBytes(Object datum) {
        if (datum == null) {
            return false;
        }
        if (super.isBytes(datum)) {
            return true;
        }
        Class c = datum.getClass();
        return c.isArray() && c.getComponentType() == Byte.TYPE;
    }

    /* JADX INFO: Access modifiers changed from: protected */
    @Override // com.flurry.org.apache.avro.generic.GenericData
    public Schema getRecordSchema(Object record) {
        return record instanceof GenericContainer ? super.getRecordSchema(record) : getSchema(record.getClass());
    }

    @Override // com.flurry.org.apache.avro.generic.GenericData
    public boolean validate(Schema schema, Object datum) {
        switch (schema.getType()) {
            case ARRAY:
                if (!datum.getClass().isArray()) {
                    return super.validate(schema, datum);
                }
                int length = Array.getLength(datum);
                for (int i = 0; i < length; i++) {
                    if (!validate(schema.getElementType(), Array.get(datum, i))) {
                        return false;
                    }
                }
                return true;
            default:
                return super.validate(schema, datum);
        }
    }

    private static Field getField(Class c, String name) {
        Map<String, Field> fields = FIELD_CACHE.get(c);
        if (fields == null) {
            fields = new ConcurrentHashMap<>();
            FIELD_CACHE.put(c, fields);
        }
        Field f = fields.get(name);
        if (f == null) {
            Field f2 = findField(c, name);
            fields.put(name, f2);
            return f2;
        }
        return f;
    }

    private static Field findField(Class original, String name) {
        Class c = original;
        do {
            try {
                Field f = c.getDeclaredField(name);
                f.setAccessible(true);
                return f;
            } catch (NoSuchFieldException e) {
                c = c.getSuperclass();
                if (c == null) {
                    throw new AvroRuntimeException("No field named " + name + " in: " + original);
                }
            }
        } while (c == null);
        throw new AvroRuntimeException("No field named " + name + " in: " + original);
    }

    /* JADX INFO: Access modifiers changed from: package-private */
    public static Class getClassProp(Schema schema, String prop) {
        String name = schema.getProp(prop);
        if (name == null) {
            return null;
        }
        try {
            return Class.forName(name);
        } catch (ClassNotFoundException e) {
            throw new AvroRuntimeException(e);
        }
    }

    /* JADX WARN: Can't fix incorrect switch cases order, some code will duplicate */
    @Override // com.flurry.org.apache.avro.specific.SpecificData
    public Class getClass(Schema schema) {
        switch (schema.getType()) {
            case ARRAY:
                Class collectionClass = getClassProp(schema, CLASS_PROP);
                return collectionClass == null ? Array.newInstance(getClass(schema.getElementType()), 0).getClass() : collectionClass;
            case STRING:
                return String.class;
            case BYTES:
                return BYTES_CLASS;
            case INT:
                String intClass = schema.getProp(CLASS_PROP);
                if (Byte.class.getName().equals(intClass)) {
                    return Byte.TYPE;
                }
                if (Short.class.getName().equals(intClass)) {
                    return Short.TYPE;
                }
                break;
        }
        return super.getClass(schema);
    }

    /* JADX INFO: Access modifiers changed from: protected */
    @Override // com.flurry.org.apache.avro.specific.SpecificData
    public Schema createSchema(Type type, Map<String, Schema> names) {
        if (type instanceof GenericArrayType) {
            Type component = ((GenericArrayType) type).getGenericComponentType();
            if (component == Byte.TYPE) {
                return Schema.create(Schema.Type.BYTES);
            }
            Schema result = Schema.createArray(createSchema(component, names));
            setElement(result, component);
            return result;
        }
        if (type instanceof ParameterizedType) {
            ParameterizedType ptype = (ParameterizedType) type;
            Class raw = (Class) ptype.getRawType();
            Type[] params = ptype.getActualTypeArguments();
            if (Map.class.isAssignableFrom(raw)) {
                Type key = params[0];
                Type value = params[1];
                if (key != String.class) {
                    throw new AvroTypeException("Map key class not String: " + key);
                }
                return Schema.createMap(createSchema(value, names));
            } else if (Collection.class.isAssignableFrom(raw)) {
                if (params.length != 1) {
                    throw new AvroTypeException("No array type specified.");
                }
                Schema schema = Schema.createArray(createSchema(params[0], names));
                schema.addProp(CLASS_PROP, raw.getName());
                return schema;
            }
        } else if (type == Byte.class || type == Byte.TYPE) {
            Schema result2 = Schema.create(Schema.Type.INT);
            result2.addProp(CLASS_PROP, Byte.class.getName());
            return result2;
        } else if (type == Short.class || type == Short.TYPE) {
            Schema result3 = Schema.create(Schema.Type.INT);
            result3.addProp(CLASS_PROP, Short.class.getName());
            return result3;
        } else if (type instanceof Class) {
            Class<?> c = (Class) type;
            if (c.isPrimitive() || Number.class.isAssignableFrom(c) || c == Void.class || c == Boolean.class) {
                return super.createSchema(type, names);
            }
            if (c.isArray()) {
                Class component2 = c.getComponentType();
                if (component2 == Byte.TYPE) {
                    return Schema.create(Schema.Type.BYTES);
                }
                Schema result4 = Schema.createArray(createSchema(component2, names));
                setElement(result4, component2);
                return result4;
            } else if (CharSequence.class.isAssignableFrom(c)) {
                return Schema.create(Schema.Type.STRING);
            } else {
                if (ByteBuffer.class.isAssignableFrom(c)) {
                    return Schema.create(Schema.Type.BYTES);
                }
                if (Collection.class.isAssignableFrom(c)) {
                    throw new AvroRuntimeException("Can't find element type of Collection");
                }
                String fullName = c.getName();
                Schema schema2 = names.get(fullName);
                if (schema2 == null) {
                    String name = c.getSimpleName();
                    String space = c.getPackage() == null ? "" : c.getPackage().getName();
                    if (c.getEnclosingClass() != null) {
                        space = c.getEnclosingClass().getName() + "$";
                    }
                    Union union = (Union) c.getAnnotation(Union.class);
                    if (union != null) {
                        return getAnnotatedUnion(union, names);
                    }
                    if (c.isAnnotationPresent(Stringable.class)) {
                        Schema result5 = Schema.create(Schema.Type.STRING);
                        result5.addProp(CLASS_PROP, c.getName());
                        return result5;
                    }
                    if (c.isEnum()) {
                        List<String> symbols = new ArrayList<>();
                        Enum[] constants = (Enum[]) c.getEnumConstants();
                        for (Enum r28 : constants) {
                            symbols.add(r28.name());
                        }
                        schema2 = Schema.createEnum(name, null, space, symbols);
                    } else if (GenericFixed.class.isAssignableFrom(c)) {
                        int size = ((FixedSize) c.getAnnotation(FixedSize.class)).value();
                        schema2 = Schema.createFixed(name, null, space, size);
                    } else if (IndexedRecord.class.isAssignableFrom(c)) {
                        return super.createSchema(type, names);
                    } else {
                        List<Schema.Field> fields = new ArrayList<>();
                        boolean error = Throwable.class.isAssignableFrom(c);
                        schema2 = Schema.createRecord(name, null, space, error);
                        names.put(c.getName(), schema2);
                        for (Field field : getFields(c)) {
                            if ((field.getModifiers() & 136) == 0) {
                                Schema fieldSchema = createFieldSchema(field, names);
                                JsonNode defaultValue = null;
                                if (fieldSchema.getType() == Schema.Type.UNION) {
                                    Schema defaultType = fieldSchema.getTypes().get(0);
                                    if (defaultType.getType() == Schema.Type.NULL) {
                                        defaultValue = NullNode.getInstance();
                                    }
                                }
                                fields.add(new Schema.Field(field.getName(), fieldSchema, null, defaultValue));
                            }
                        }
                        if (error) {
                            fields.add(new Schema.Field("detailMessage", THROWABLE_MESSAGE, null, null));
                        }
                        schema2.setFields(fields);
                    }
                    names.put(fullName, schema2);
                }
                return schema2;
            }
        }
        return super.createSchema(type, names);
    }

    private void setElement(Schema schema, Type element) {
        if (element instanceof Class) {
            Class<?> c = (Class) element;
            Union union = (Union) c.getAnnotation(Union.class);
            if (union != null) {
                schema.addProp(ELEMENT_PROP, c.getName());
            }
        }
    }

    private Schema getAnnotatedUnion(Union union, Map<String, Schema> names) {
        List<Schema> branches = new ArrayList<>();
        Class[] arr$ = union.value();
        for (Class branch : arr$) {
            branches.add(createSchema(branch, names));
        }
        return Schema.createUnion(branches);
    }

    public static Schema makeNullable(Schema schema) {
        return Schema.createUnion(Arrays.asList(Schema.create(Schema.Type.NULL), schema));
    }

    private Collection<Field> getFields(Class recordClass) {
        Map<String, Field> fields = new LinkedHashMap<>();
        Class c = recordClass;
        do {
            if (c.getPackage() != null && c.getPackage().getName().startsWith("java.")) {
                break;
            }
            Field[] arr$ = c.getDeclaredFields();
            for (Field field : arr$) {
                if ((field.getModifiers() & 136) == 0 && fields.put(field.getName(), field) != null) {
                    throw new AvroTypeException(c + " contains two fields named: " + field);
                }
            }
            c = c.getSuperclass();
        } while (c != null);
        return fields.values();
    }

    protected Schema createFieldSchema(Field field, Map<String, Schema> names) {
        Schema schema = createSchema(field.getGenericType(), names);
        if (field.isAnnotationPresent(Nullable.class)) {
            return makeNullable(schema);
        }
        return schema;
    }

    @Override // com.flurry.org.apache.avro.specific.SpecificData
    public Protocol getProtocol(Class iface) {
        Protocol protocol = new Protocol(iface.getSimpleName(), iface.getPackage() == null ? "" : iface.getPackage().getName());
        Map<String, Schema> names = new LinkedHashMap<>();
        Map<String, Protocol.Message> messages = protocol.getMessages();
        Method[] arr$ = iface.getMethods();
        for (Method method : arr$) {
            if ((method.getModifiers() & 8) == 0) {
                String name = method.getName();
                if (messages.containsKey(name)) {
                    throw new AvroTypeException("Two methods with same name: " + name);
                }
                messages.put(name, getMessage(method, protocol, names));
            }
        }
        List<Schema> types = new ArrayList<>();
        types.addAll(names.values());
        Collections.reverse(types);
        protocol.setTypes(types);
        return protocol;
    }

    private Protocol.Message getMessage(Method method, Protocol protocol, Map<String, Schema> names) {
        List<Schema.Field> fields = new ArrayList<>();
        String[] paramNames = this.paranamer.lookupParameterNames(method);
        Type[] paramTypes = method.getGenericParameterTypes();
        Annotation[][] annotations = method.getParameterAnnotations();
        for (int i = 0; i < paramTypes.length; i++) {
            Schema paramSchema = getSchema(paramTypes[i], names);
            for (int j = 0; j < annotations[i].length; j++) {
                if (annotations[i][j] instanceof Union) {
                    paramSchema = getAnnotatedUnion((Union) annotations[i][j], names);
                } else if (annotations[i][j] instanceof Nullable) {
                    paramSchema = makeNullable(paramSchema);
                }
            }
            String paramName = paramNames.length == paramTypes.length ? paramNames[i] : paramSchema.getName() + i;
            fields.add(new Schema.Field(paramName, paramSchema, null, null));
        }
        Schema request = Schema.createRecord(fields);
        Union union = (Union) method.getAnnotation(Union.class);
        Schema response = union == null ? getSchema(method.getGenericReturnType(), names) : getAnnotatedUnion(union, names);
        if (method.isAnnotationPresent(Nullable.class)) {
            response = makeNullable(response);
        }
        List<Schema> errs = new ArrayList<>();
        errs.add(Protocol.SYSTEM_ERROR);
        Type[] arr$ = method.getGenericExceptionTypes();
        for (Type err : arr$) {
            if (err != AvroRemoteException.class) {
                errs.add(getSchema(err, names));
            }
        }
        Schema errors = Schema.createUnion(errs);
        return protocol.createMessage(method.getName(), null, request, response, errors);
    }

    private Schema getSchema(Type type, Map<String, Schema> names) {
        try {
            return createSchema(type, names);
        } catch (AvroTypeException e) {
            throw new AvroTypeException("Error getting schema for " + type + ": " + e.getMessage(), e);
        }
    }

    /* JADX INFO: Access modifiers changed from: protected */
    @Override // com.flurry.org.apache.avro.specific.SpecificData, com.flurry.org.apache.avro.generic.GenericData
    public int compare(Object o1, Object o2, Schema s, boolean equals) {
        switch (s.getType()) {
            case ARRAY:
                if (o1.getClass().isArray()) {
                    Schema elementType = s.getElementType();
                    int l1 = Array.getLength(o1);
                    int l2 = Array.getLength(o2);
                    int l = Math.min(l1, l2);
                    for (int i = 0; i < l; i++) {
                        int compare = compare(Array.get(o1, i), Array.get(o2, i), elementType, equals);
                        if (compare != 0) {
                            return compare;
                        }
                    }
                    return l1 - l2;
                }
                break;
            case BYTES:
                if (o1.getClass().isArray()) {
                    byte[] b1 = (byte[]) o1;
                    byte[] b2 = (byte[]) o2;
                    return BinaryData.compareBytes(b1, 0, b1.length, b2, 0, b2.length);
                }
                break;
        }
        return super.compare(o1, o2, s, equals);
    }
}
