package com.flurry.org.apache.avro.io.parsing;

import com.flurry.org.apache.avro.Schema;
import com.flurry.org.apache.avro.file.DataFileConstants;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;

/* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
public abstract class Symbol {
    public final Kind kind;
    public final Symbol[] production;
    public static final Symbol NULL = new Terminal(DataFileConstants.NULL_CODEC);
    public static final Symbol BOOLEAN = new Terminal("boolean");
    public static final Symbol INT = new Terminal("int");
    public static final Symbol LONG = new Terminal("long");
    public static final Symbol FLOAT = new Terminal("float");
    public static final Symbol DOUBLE = new Terminal("double");
    public static final Symbol STRING = new Terminal("string");
    public static final Symbol BYTES = new Terminal("bytes");
    public static final Symbol FIXED = new Terminal("fixed");
    public static final Symbol ENUM = new Terminal("enum");
    public static final Symbol UNION = new Terminal("union");
    public static final Symbol ARRAY_START = new Terminal("array-start");
    public static final Symbol ARRAY_END = new Terminal("array-end");
    public static final Symbol MAP_START = new Terminal("map-start");
    public static final Symbol MAP_END = new Terminal("map-end");
    public static final Symbol ITEM_END = new Terminal("item-end");
    public static final Symbol FIELD_ACTION = new Terminal("field-action");
    public static final Symbol RECORD_START = new ImplicitAction(false);
    public static final Symbol RECORD_END = new ImplicitAction(true);
    public static final Symbol UNION_END = new ImplicitAction(true);
    public static final Symbol FIELD_END = new ImplicitAction(true);
    public static final Symbol DEFAULT_END_ACTION = new ImplicitAction(true);
    public static final Symbol MAP_KEY_MARKER = new Terminal("map-key-marker");

    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public enum Kind {
        TERMINAL,
        ROOT,
        SEQUENCE,
        REPEATER,
        ALTERNATIVE,
        IMPLICIT_ACTION,
        EXPLICIT_ACTION
    }

    protected Symbol(Kind kind) {
        this(kind, null);
    }

    protected Symbol(Kind kind, Symbol[] production) {
        this.production = production;
        this.kind = kind;
    }

    /* JADX INFO: Access modifiers changed from: package-private */
    public static Symbol root(Symbol... symbols) {
        return new Root(symbols);
    }

    /* JADX INFO: Access modifiers changed from: package-private */
    public static Symbol seq(Symbol... production) {
        return new Sequence(production);
    }

    /* JADX INFO: Access modifiers changed from: package-private */
    public static Symbol repeat(Symbol endSymbol, Symbol... symsToRepeat) {
        return new Repeater(symsToRepeat);
    }

    /* JADX INFO: Access modifiers changed from: package-private */
    public static Symbol alt(Symbol[] symbols, String[] labels) {
        return new Alternative(symbols, labels);
    }

    /* JADX INFO: Access modifiers changed from: package-private */
    public static Symbol error(String e) {
        return new ErrorAction(e);
    }

    /* JADX INFO: Access modifiers changed from: package-private */
    public static Symbol resolve(Symbol w, Symbol r) {
        return new ResolvingAction(r);
    }

    /* JADX INFO: Access modifiers changed from: private */
    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public static class Fixup {
        public final int pos;
        public final Symbol[] symbols;

        public Fixup(Symbol[] symbols, int pos) {
            this.symbols = symbols;
            this.pos = pos;
        }
    }

    public Symbol flatten(Map<Sequence, Sequence> map, Map<Sequence, List<Fixup>> map2) {
        return this;
    }

    public int flattenedSize() {
        return 1;
    }

    static void flatten(Symbol[] in, int start, Symbol[] out, int skip, Map<Sequence, Sequence> map, Map<Sequence, List<Fixup>> map2) {
        int j = skip;
        for (int i = start; i < in.length; i++) {
            Symbol s = in[i].flatten(map, map2);
            if (s instanceof Sequence) {
                Symbol[] p = s.production;
                List<Fixup> l = map2.get(s);
                if (l == null) {
                    System.arraycopy(p, 0, out, j, p.length);
                } else {
                    l.add(new Fixup(out, j));
                }
                j += p.length;
            } else {
                out[j] = s;
                j++;
            }
        }
    }

    protected static int flattenedSize(Symbol[] symbols, int start) {
        int result = 0;
        for (int i = start; i < symbols.length; i++) {
            if (symbols[i] instanceof Sequence) {
                Sequence s = (Sequence) symbols[i];
                result += s.flattenedSize();
            } else {
                result++;
            }
        }
        return result;
    }

    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    private static class Terminal extends Symbol {
        private final String printName;

        public Terminal(String printName) {
            super(Kind.TERMINAL);
            this.printName = printName;
        }

        public String toString() {
            return this.printName;
        }
    }

    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public static class ImplicitAction extends Symbol {
        public final boolean isTrailing;

        private ImplicitAction() {
            this(false);
        }

        private ImplicitAction(boolean isTrailing) {
            super(Kind.IMPLICIT_ACTION);
            this.isTrailing = isTrailing;
        }
    }

    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    protected static class Root extends Symbol {
        private Root(Symbol... symbols) {
            super(Kind.ROOT, makeProduction(symbols));
            this.production[0] = this;
        }

        private static Symbol[] makeProduction(Symbol[] symbols) {
            Symbol[] result = new Symbol[flattenedSize(symbols, 0) + 1];
            flatten(symbols, 0, result, 1, new HashMap(), new HashMap());
            return result;
        }
    }

    /* JADX INFO: Access modifiers changed from: protected */
    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public static class Sequence extends Symbol implements Iterable<Symbol> {
        @Override // com.flurry.org.apache.avro.io.parsing.Symbol
        public /* bridge */ /* synthetic */ Symbol flatten(Map x0, Map x1) {
            return flatten((Map<Sequence, Sequence>) x0, (Map<Sequence, List<Fixup>>) x1);
        }

        private Sequence(Symbol[] productions) {
            super(Kind.SEQUENCE, productions);
        }

        public Symbol get(int index) {
            return this.production[index];
        }

        public int size() {
            return this.production.length;
        }

        @Override // java.lang.Iterable
        public Iterator<Symbol> iterator() {
            return new Iterator<Symbol>() { // from class: com.flurry.org.apache.avro.io.parsing.Symbol.Sequence.1
                private int pos;

                {
                    this.pos = Sequence.this.production.length;
                }

                @Override // java.util.Iterator
                public boolean hasNext() {
                    return this.pos > 0;
                }

                /* JADX WARN: Can't rename method to resolve collision */
                @Override // java.util.Iterator
                public Symbol next() {
                    if (this.pos > 0) {
                        Symbol[] symbolArr = Sequence.this.production;
                        int i = this.pos - 1;
                        this.pos = i;
                        return symbolArr[i];
                    }
                    throw new NoSuchElementException();
                }

                @Override // java.util.Iterator
                public void remove() {
                    throw new UnsupportedOperationException();
                }
            };
        }

        @Override // com.flurry.org.apache.avro.io.parsing.Symbol
        public Sequence flatten(Map<Sequence, Sequence> map, Map<Sequence, List<Fixup>> map2) {
            Sequence result = map.get(this);
            if (result == null) {
                result = new Sequence(new Symbol[flattenedSize()]);
                map.put(this, result);
                List<Fixup> l = new ArrayList<>();
                map2.put(result, l);
                flatten(this.production, 0, result.production, 0, map, map2);
                for (Fixup f : l) {
                    System.arraycopy(result.production, 0, f.symbols, f.pos, result.production.length);
                }
                map2.remove(result);
            }
            return result;
        }

        @Override // com.flurry.org.apache.avro.io.parsing.Symbol
        public final int flattenedSize() {
            return flattenedSize(this.production, 0);
        }
    }

    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public static class Repeater extends Symbol {
        public final Symbol end;

        @Override // com.flurry.org.apache.avro.io.parsing.Symbol
        public /* bridge */ /* synthetic */ Symbol flatten(Map x0, Map x1) {
            return flatten((Map<Sequence, Sequence>) x0, (Map<Sequence, List<Fixup>>) x1);
        }

        private Repeater(Symbol end, Symbol... sequenceToRepeat) {
            super(Kind.REPEATER, makeProduction(sequenceToRepeat));
            this.end = end;
            this.production[0] = this;
        }

        private static Symbol[] makeProduction(Symbol[] p) {
            Symbol[] result = new Symbol[p.length + 1];
            System.arraycopy(p, 0, result, 1, p.length);
            return result;
        }

        @Override // com.flurry.org.apache.avro.io.parsing.Symbol
        public Repeater flatten(Map<Sequence, Sequence> map, Map<Sequence, List<Fixup>> map2) {
            Repeater result = new Repeater(this.end, new Symbol[flattenedSize(this.production, 1)]);
            flatten(this.production, 1, result.production, 1, map, map2);
            return result;
        }
    }

    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public static class Alternative extends Symbol {
        public final String[] labels;
        public final Symbol[] symbols;

        @Override // com.flurry.org.apache.avro.io.parsing.Symbol
        public /* bridge */ /* synthetic */ Symbol flatten(Map x0, Map x1) {
            return flatten((Map<Sequence, Sequence>) x0, (Map<Sequence, List<Fixup>>) x1);
        }

        private Alternative(Symbol[] symbols, String[] labels) {
            super(Kind.ALTERNATIVE);
            this.symbols = symbols;
            this.labels = labels;
        }

        public Symbol getSymbol(int index) {
            return this.symbols[index];
        }

        public String getLabel(int index) {
            return this.labels[index];
        }

        public int size() {
            return this.symbols.length;
        }

        public int findLabel(String label) {
            if (label != null) {
                for (int i = 0; i < this.labels.length; i++) {
                    if (label.equals(this.labels[i])) {
                        return i;
                    }
                }
            }
            return -1;
        }

        @Override // com.flurry.org.apache.avro.io.parsing.Symbol
        public Alternative flatten(Map<Sequence, Sequence> map, Map<Sequence, List<Fixup>> map2) {
            Symbol[] ss = new Symbol[this.symbols.length];
            for (int i = 0; i < ss.length; i++) {
                ss[i] = this.symbols[i].flatten(map, map2);
            }
            return new Alternative(ss, this.labels);
        }
    }

    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public static class ErrorAction extends ImplicitAction {
        public final String msg;

        private ErrorAction(String msg) {
            super();
            this.msg = msg;
        }
    }

    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public static class IntCheckAction extends Symbol {
        public final int size;

        public IntCheckAction(int size) {
            super(Kind.EXPLICIT_ACTION);
            this.size = size;
        }
    }

    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public static class EnumAdjustAction extends IntCheckAction {
        public final Object[] adjustments;

        public EnumAdjustAction(int rsymCount, Object[] adjustments) {
            super(rsymCount);
            this.adjustments = adjustments;
        }
    }

    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public static class WriterUnionAction extends ImplicitAction {
        public WriterUnionAction() {
            super();
        }
    }

    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public static class ResolvingAction extends ImplicitAction {
        public final Symbol reader;
        public final Symbol writer;

        @Override // com.flurry.org.apache.avro.io.parsing.Symbol
        public /* bridge */ /* synthetic */ Symbol flatten(Map x0, Map x1) {
            return flatten((Map<Sequence, Sequence>) x0, (Map<Sequence, List<Fixup>>) x1);
        }

        private ResolvingAction(Symbol writer, Symbol reader) {
            super();
            this.writer = writer;
            this.reader = reader;
        }

        @Override // com.flurry.org.apache.avro.io.parsing.Symbol
        public ResolvingAction flatten(Map<Sequence, Sequence> map, Map<Sequence, List<Fixup>> map2) {
            return new ResolvingAction(this.writer.flatten(map, map2), this.reader.flatten(map, map2));
        }
    }

    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public static class SkipAction extends ImplicitAction {
        public final Symbol symToSkip;

        @Override // com.flurry.org.apache.avro.io.parsing.Symbol
        public /* bridge */ /* synthetic */ Symbol flatten(Map x0, Map x1) {
            return flatten((Map<Sequence, Sequence>) x0, (Map<Sequence, List<Fixup>>) x1);
        }

        public SkipAction(Symbol symToSkip) {
            super(true);
            this.symToSkip = symToSkip;
        }

        @Override // com.flurry.org.apache.avro.io.parsing.Symbol
        public SkipAction flatten(Map<Sequence, Sequence> map, Map<Sequence, List<Fixup>> map2) {
            return new SkipAction(this.symToSkip.flatten(map, map2));
        }
    }

    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public static class FieldAdjustAction extends ImplicitAction {
        public final String fname;
        public final int rindex;

        public FieldAdjustAction(int rindex, String fname) {
            super();
            this.rindex = rindex;
            this.fname = fname;
        }
    }

    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public static final class FieldOrderAction extends ImplicitAction {
        public final Schema.Field[] fields;

        public FieldOrderAction(Schema.Field[] fields) {
            super();
            this.fields = fields;
        }
    }

    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public static class DefaultStartAction extends ImplicitAction {
        public final byte[] contents;

        public DefaultStartAction(byte[] contents) {
            super();
            this.contents = contents;
        }
    }

    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public static class UnionAdjustAction extends ImplicitAction {
        public final int rindex;
        public final Symbol symToParse;

        @Override // com.flurry.org.apache.avro.io.parsing.Symbol
        public /* bridge */ /* synthetic */ Symbol flatten(Map x0, Map x1) {
            return flatten((Map<Sequence, Sequence>) x0, (Map<Sequence, List<Fixup>>) x1);
        }

        public UnionAdjustAction(int rindex, Symbol symToParse) {
            super();
            this.rindex = rindex;
            this.symToParse = symToParse;
        }

        @Override // com.flurry.org.apache.avro.io.parsing.Symbol
        public UnionAdjustAction flatten(Map<Sequence, Sequence> map, Map<Sequence, List<Fixup>> map2) {
            return new UnionAdjustAction(this.rindex, this.symToParse.flatten(map, map2));
        }
    }

    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public static class EnumLabelsAction extends IntCheckAction {
        public final List<String> symbols;

        public EnumLabelsAction(List<String> symbols) {
            super(symbols.size());
            this.symbols = symbols;
        }

        public String getLabel(int n) {
            return this.symbols.get(n);
        }

        public int findLabel(String l) {
            if (l != null) {
                for (int i = 0; i < this.symbols.size(); i++) {
                    if (l.equals(this.symbols.get(i))) {
                        return i;
                    }
                }
            }
            return -1;
        }
    }
}
