package com.flurry.org.apache.avro.io;

import com.flurry.org.apache.avro.AvroTypeException;
import com.flurry.org.apache.avro.Schema;
import com.flurry.org.apache.avro.io.parsing.Parser;
import com.flurry.org.apache.avro.io.parsing.Symbol;
import com.flurry.org.apache.avro.io.parsing.ValidatingGrammarGenerator;
import com.flurry.org.apache.avro.util.Utf8;
import java.io.IOException;
import java.nio.ByteBuffer;

/* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
public class ValidatingEncoder extends ParsingEncoder implements Parser.ActionHandler {
    protected Encoder out;
    protected final Parser parser;

    ValidatingEncoder(Symbol root, Encoder out) throws IOException {
        this.out = out;
        this.parser = new Parser(root, this);
    }

    /* JADX INFO: Access modifiers changed from: package-private */
    public ValidatingEncoder(Schema schema, Encoder in) throws IOException {
        this(new ValidatingGrammarGenerator().generate(schema), in);
    }

    @Override // java.io.Flushable
    public void flush() throws IOException {
        this.out.flush();
    }

    public ValidatingEncoder configure(Encoder encoder) {
        this.parser.reset();
        this.out = encoder;
        return this;
    }

    @Override // com.flurry.org.apache.avro.io.Encoder
    public void writeNull() throws IOException {
        this.parser.advance(Symbol.NULL);
        this.out.writeNull();
    }

    @Override // com.flurry.org.apache.avro.io.Encoder
    public void writeBoolean(boolean b) throws IOException {
        this.parser.advance(Symbol.BOOLEAN);
        this.out.writeBoolean(b);
    }

    @Override // com.flurry.org.apache.avro.io.Encoder
    public void writeInt(int n) throws IOException {
        this.parser.advance(Symbol.INT);
        this.out.writeInt(n);
    }

    @Override // com.flurry.org.apache.avro.io.Encoder
    public void writeLong(long n) throws IOException {
        this.parser.advance(Symbol.LONG);
        this.out.writeLong(n);
    }

    @Override // com.flurry.org.apache.avro.io.Encoder
    public void writeFloat(float f) throws IOException {
        this.parser.advance(Symbol.FLOAT);
        this.out.writeFloat(f);
    }

    @Override // com.flurry.org.apache.avro.io.Encoder
    public void writeDouble(double d) throws IOException {
        this.parser.advance(Symbol.DOUBLE);
        this.out.writeDouble(d);
    }

    @Override // com.flurry.org.apache.avro.io.Encoder
    public void writeString(Utf8 utf8) throws IOException {
        this.parser.advance(Symbol.STRING);
        this.out.writeString(utf8);
    }

    @Override // com.flurry.org.apache.avro.io.Encoder
    public void writeString(String str) throws IOException {
        this.parser.advance(Symbol.STRING);
        this.out.writeString(str);
    }

    @Override // com.flurry.org.apache.avro.io.Encoder
    public void writeString(CharSequence charSequence) throws IOException {
        this.parser.advance(Symbol.STRING);
        this.out.writeString(charSequence);
    }

    @Override // com.flurry.org.apache.avro.io.Encoder
    public void writeBytes(ByteBuffer bytes) throws IOException {
        this.parser.advance(Symbol.BYTES);
        this.out.writeBytes(bytes);
    }

    @Override // com.flurry.org.apache.avro.io.Encoder
    public void writeBytes(byte[] bytes, int start, int len) throws IOException {
        this.parser.advance(Symbol.BYTES);
        this.out.writeBytes(bytes, start, len);
    }

    @Override // com.flurry.org.apache.avro.io.Encoder
    public void writeFixed(byte[] bytes, int start, int len) throws IOException {
        this.parser.advance(Symbol.FIXED);
        Symbol.IntCheckAction top = (Symbol.IntCheckAction) this.parser.popSymbol();
        if (len != top.size) {
            throw new AvroTypeException("Incorrect length for fixed binary: expected " + top.size + " but received " + len + " bytes.");
        }
        this.out.writeFixed(bytes, start, len);
    }

    @Override // com.flurry.org.apache.avro.io.Encoder
    public void writeEnum(int e) throws IOException {
        this.parser.advance(Symbol.ENUM);
        Symbol.IntCheckAction top = (Symbol.IntCheckAction) this.parser.popSymbol();
        if (e < 0 || e >= top.size) {
            throw new AvroTypeException("Enumeration out of range: max is " + top.size + " but received " + e);
        }
        this.out.writeEnum(e);
    }

    @Override // com.flurry.org.apache.avro.io.Encoder
    public void writeArrayStart() throws IOException {
        push();
        this.parser.advance(Symbol.ARRAY_START);
        this.out.writeArrayStart();
    }

    @Override // com.flurry.org.apache.avro.io.Encoder
    public void writeArrayEnd() throws IOException {
        this.parser.advance(Symbol.ARRAY_END);
        this.out.writeArrayEnd();
        pop();
    }

    @Override // com.flurry.org.apache.avro.io.Encoder
    public void writeMapStart() throws IOException {
        push();
        this.parser.advance(Symbol.MAP_START);
        this.out.writeMapStart();
    }

    @Override // com.flurry.org.apache.avro.io.Encoder
    public void writeMapEnd() throws IOException {
        this.parser.advance(Symbol.MAP_END);
        this.out.writeMapEnd();
        pop();
    }

    @Override // com.flurry.org.apache.avro.io.ParsingEncoder, com.flurry.org.apache.avro.io.Encoder
    public void setItemCount(long itemCount) throws IOException {
        super.setItemCount(itemCount);
        this.out.setItemCount(itemCount);
    }

    @Override // com.flurry.org.apache.avro.io.ParsingEncoder, com.flurry.org.apache.avro.io.Encoder
    public void startItem() throws IOException {
        super.startItem();
        this.out.startItem();
    }

    @Override // com.flurry.org.apache.avro.io.Encoder
    public void writeIndex(int unionIndex) throws IOException {
        this.parser.advance(Symbol.UNION);
        Symbol.Alternative top = (Symbol.Alternative) this.parser.popSymbol();
        this.parser.pushSymbol(top.getSymbol(unionIndex));
        this.out.writeIndex(unionIndex);
    }

    @Override // com.flurry.org.apache.avro.io.parsing.Parser.ActionHandler
    public Symbol doAction(Symbol input, Symbol top) throws IOException {
        return null;
    }
}
