package com.flurry.org.apache.avro.io;

import com.flurry.org.apache.avro.AvroTypeException;
import com.flurry.org.apache.avro.Schema;
import com.flurry.org.apache.avro.io.parsing.Parser;
import com.flurry.org.apache.avro.io.parsing.Symbol;
import com.flurry.org.apache.avro.io.parsing.ValidatingGrammarGenerator;
import com.flurry.org.apache.avro.util.Utf8;
import java.io.IOException;
import java.nio.ByteBuffer;

/* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
public class ValidatingDecoder extends ParsingDecoder implements Parser.ActionHandler {
    protected Decoder in;

    /* JADX INFO: Access modifiers changed from: package-private */
    public ValidatingDecoder(Symbol root, Decoder in) throws IOException {
        super(root);
        configure(in);
    }

    /* JADX INFO: Access modifiers changed from: package-private */
    public ValidatingDecoder(Schema schema, Decoder in) throws IOException {
        this(getSymbol(schema), in);
    }

    private static Symbol getSymbol(Schema schema) {
        if (schema == null) {
            throw new NullPointerException("Schema cannot be null");
        }
        return new ValidatingGrammarGenerator().generate(schema);
    }

    public ValidatingDecoder configure(Decoder in) throws IOException {
        this.parser.reset();
        this.in = in;
        return this;
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public void readNull() throws IOException {
        this.parser.advance(Symbol.NULL);
        this.in.readNull();
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public boolean readBoolean() throws IOException {
        this.parser.advance(Symbol.BOOLEAN);
        return this.in.readBoolean();
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public int readInt() throws IOException {
        this.parser.advance(Symbol.INT);
        return this.in.readInt();
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public long readLong() throws IOException {
        this.parser.advance(Symbol.LONG);
        return this.in.readLong();
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public float readFloat() throws IOException {
        this.parser.advance(Symbol.FLOAT);
        return this.in.readFloat();
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public double readDouble() throws IOException {
        this.parser.advance(Symbol.DOUBLE);
        return this.in.readDouble();
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public Utf8 readString(Utf8 old) throws IOException {
        this.parser.advance(Symbol.STRING);
        return this.in.readString(old);
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public String readString() throws IOException {
        this.parser.advance(Symbol.STRING);
        return this.in.readString();
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public void skipString() throws IOException {
        this.parser.advance(Symbol.STRING);
        this.in.skipString();
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public ByteBuffer readBytes(ByteBuffer old) throws IOException {
        this.parser.advance(Symbol.BYTES);
        return this.in.readBytes(old);
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public void skipBytes() throws IOException {
        this.parser.advance(Symbol.BYTES);
        this.in.skipBytes();
    }

    private void checkFixed(int size) throws IOException {
        this.parser.advance(Symbol.FIXED);
        Symbol.IntCheckAction top = (Symbol.IntCheckAction) this.parser.popSymbol();
        if (size != top.size) {
            throw new AvroTypeException("Incorrect length for fixed binary: expected " + top.size + " but received " + size + " bytes.");
        }
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public void readFixed(byte[] bytes, int start, int len) throws IOException {
        checkFixed(len);
        this.in.readFixed(bytes, start, len);
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public void skipFixed(int length) throws IOException {
        checkFixed(length);
        this.in.skipFixed(length);
    }

    @Override // com.flurry.org.apache.avro.io.ParsingDecoder
    protected void skipFixed() throws IOException {
        this.parser.advance(Symbol.FIXED);
        Symbol.IntCheckAction top = (Symbol.IntCheckAction) this.parser.popSymbol();
        this.in.skipFixed(top.size);
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public int readEnum() throws IOException {
        this.parser.advance(Symbol.ENUM);
        Symbol.IntCheckAction top = (Symbol.IntCheckAction) this.parser.popSymbol();
        int result = this.in.readEnum();
        if (result < 0 || result >= top.size) {
            throw new AvroTypeException("Enumeration out of range: max is " + top.size + " but received " + result);
        }
        return result;
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public long readArrayStart() throws IOException {
        this.parser.advance(Symbol.ARRAY_START);
        long result = this.in.readArrayStart();
        if (result == 0) {
            this.parser.advance(Symbol.ARRAY_END);
        }
        return result;
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public long arrayNext() throws IOException {
        this.parser.processTrailingImplicitActions();
        long result = this.in.arrayNext();
        if (result == 0) {
            this.parser.advance(Symbol.ARRAY_END);
        }
        return result;
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public long skipArray() throws IOException {
        this.parser.advance(Symbol.ARRAY_START);
        long c = this.in.skipArray();
        while (c != 0) {
            while (true) {
                long c2 = c;
                c = c2 - 1;
                if (c2 > 0) {
                    this.parser.skipRepeater();
                }
            }
            c = this.in.skipArray();
        }
        this.parser.advance(Symbol.ARRAY_END);
        return 0L;
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public long readMapStart() throws IOException {
        this.parser.advance(Symbol.MAP_START);
        long result = this.in.readMapStart();
        if (result == 0) {
            this.parser.advance(Symbol.MAP_END);
        }
        return result;
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public long mapNext() throws IOException {
        this.parser.processTrailingImplicitActions();
        long result = this.in.mapNext();
        if (result == 0) {
            this.parser.advance(Symbol.MAP_END);
        }
        return result;
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public long skipMap() throws IOException {
        this.parser.advance(Symbol.MAP_START);
        long c = this.in.skipMap();
        while (c != 0) {
            while (true) {
                long c2 = c;
                c = c2 - 1;
                if (c2 > 0) {
                    this.parser.skipRepeater();
                }
            }
            c = this.in.skipMap();
        }
        this.parser.advance(Symbol.MAP_END);
        return 0L;
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public int readIndex() throws IOException {
        this.parser.advance(Symbol.UNION);
        Symbol.Alternative top = (Symbol.Alternative) this.parser.popSymbol();
        int result = this.in.readIndex();
        this.parser.pushSymbol(top.getSymbol(result));
        return result;
    }

    public Symbol doAction(Symbol input, Symbol top) throws IOException {
        return null;
    }
}
