package com.flurry.org.apache.avro.io;

import com.flurry.org.apache.avro.AvroTypeException;
import com.flurry.org.apache.avro.Schema;
import com.flurry.org.apache.avro.file.DataFileConstants;
import com.flurry.org.apache.avro.io.parsing.JsonGrammarGenerator;
import com.flurry.org.apache.avro.io.parsing.Parser;
import com.flurry.org.apache.avro.io.parsing.Symbol;
import com.flurry.org.apache.avro.util.Utf8;
import com.flurry.org.codehaus.jackson.Base64Variant;
import com.flurry.org.codehaus.jackson.JsonFactory;
import com.flurry.org.codehaus.jackson.JsonLocation;
import com.flurry.org.codehaus.jackson.JsonParser;
import com.flurry.org.codehaus.jackson.JsonStreamContext;
import com.flurry.org.codehaus.jackson.JsonToken;
import com.flurry.org.codehaus.jackson.ObjectCodec;
import com.prime31.util.IabHelper;
import com.tapjoy.TapjoyVideoObject;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Stack;

/* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
public class JsonDecoder extends ParsingDecoder implements Parser.ActionHandler {
    static final String CHARSET = "ISO-8859-1";
    private static JsonFactory jsonFactory = new JsonFactory();
    ReorderBuffer currentReorderBuffer;
    private JsonParser in;
    Stack<ReorderBuffer> reorderBuffers;

    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    private static class ReorderBuffer {
        public JsonParser origParser;
        public Map<String, List<JsonElement>> savedFields;

        private ReorderBuffer() {
            this.savedFields = new HashMap();
            this.origParser = null;
        }
    }

    private JsonDecoder(Symbol root, InputStream in) throws IOException {
        super(root);
        this.reorderBuffers = new Stack<>();
        configure(in);
    }

    private JsonDecoder(Symbol root, String in) throws IOException {
        super(root);
        this.reorderBuffers = new Stack<>();
        configure(in);
    }

    /* JADX INFO: Access modifiers changed from: package-private */
    public JsonDecoder(Schema schema, InputStream in) throws IOException {
        this(getSymbol(schema), in);
    }

    /* JADX INFO: Access modifiers changed from: package-private */
    public JsonDecoder(Schema schema, String in) throws IOException {
        this(getSymbol(schema), in);
    }

    private static Symbol getSymbol(Schema schema) {
        if (schema == null) {
            throw new NullPointerException("Schema cannot be null!");
        }
        return new JsonGrammarGenerator().generate(schema);
    }

    public JsonDecoder configure(InputStream in) throws IOException {
        if (in == null) {
            throw new NullPointerException("InputStream to read from cannot be null!");
        }
        this.parser.reset();
        this.in = jsonFactory.createJsonParser(in);
        this.in.nextToken();
        return this;
    }

    public JsonDecoder configure(String in) throws IOException {
        if (in == null) {
            throw new NullPointerException("String to read from cannot be null!");
        }
        this.parser.reset();
        this.in = new JsonFactory().createJsonParser(in);
        this.in.nextToken();
        return this;
    }

    private void advance(Symbol symbol) throws IOException {
        this.parser.processTrailingImplicitActions();
        if (this.in.getCurrentToken() == null && this.parser.depth() == 1) {
            throw new EOFException();
        }
        this.parser.advance(symbol);
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public void readNull() throws IOException {
        advance(Symbol.NULL);
        if (this.in.getCurrentToken() == JsonToken.VALUE_NULL) {
            this.in.nextToken();
            return;
        }
        throw error(DataFileConstants.NULL_CODEC);
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public boolean readBoolean() throws IOException {
        advance(Symbol.BOOLEAN);
        JsonToken t = this.in.getCurrentToken();
        if (t == JsonToken.VALUE_TRUE || t == JsonToken.VALUE_FALSE) {
            this.in.nextToken();
            return t == JsonToken.VALUE_TRUE;
        }
        throw error("boolean");
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public int readInt() throws IOException {
        advance(Symbol.INT);
        if (this.in.getCurrentToken() == JsonToken.VALUE_NUMBER_INT) {
            int result = this.in.getIntValue();
            this.in.nextToken();
            return result;
        }
        throw error("int");
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public long readLong() throws IOException {
        advance(Symbol.LONG);
        if (this.in.getCurrentToken() == JsonToken.VALUE_NUMBER_INT) {
            long result = this.in.getLongValue();
            this.in.nextToken();
            return result;
        }
        throw error("long");
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public float readFloat() throws IOException {
        advance(Symbol.FLOAT);
        if (this.in.getCurrentToken() == JsonToken.VALUE_NUMBER_FLOAT) {
            float result = this.in.getFloatValue();
            this.in.nextToken();
            return result;
        }
        throw error("float");
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public double readDouble() throws IOException {
        advance(Symbol.DOUBLE);
        if (this.in.getCurrentToken() == JsonToken.VALUE_NUMBER_FLOAT) {
            double result = this.in.getDoubleValue();
            this.in.nextToken();
            return result;
        }
        throw error("double");
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public Utf8 readString(Utf8 old) throws IOException {
        return new Utf8(readString());
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public String readString() throws IOException {
        advance(Symbol.STRING);
        if (this.parser.topSymbol() == Symbol.MAP_KEY_MARKER) {
            this.parser.advance(Symbol.MAP_KEY_MARKER);
            if (this.in.getCurrentToken() != JsonToken.FIELD_NAME) {
                throw error("map-key");
            }
        } else if (this.in.getCurrentToken() != JsonToken.VALUE_STRING) {
            throw error("string");
        }
        String result = this.in.getText();
        this.in.nextToken();
        return result;
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public void skipString() throws IOException {
        advance(Symbol.STRING);
        if (this.parser.topSymbol() == Symbol.MAP_KEY_MARKER) {
            this.parser.advance(Symbol.MAP_KEY_MARKER);
            if (this.in.getCurrentToken() != JsonToken.FIELD_NAME) {
                throw error("map-key");
            }
        } else if (this.in.getCurrentToken() != JsonToken.VALUE_STRING) {
            throw error("string");
        }
        this.in.nextToken();
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public ByteBuffer readBytes(ByteBuffer old) throws IOException {
        advance(Symbol.BYTES);
        if (this.in.getCurrentToken() == JsonToken.VALUE_STRING) {
            byte[] result = readByteArray();
            this.in.nextToken();
            return ByteBuffer.wrap(result);
        }
        throw error("bytes");
    }

    private byte[] readByteArray() throws IOException {
        byte[] result = this.in.getText().getBytes(CHARSET);
        return result;
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public void skipBytes() throws IOException {
        advance(Symbol.BYTES);
        if (this.in.getCurrentToken() == JsonToken.VALUE_STRING) {
            this.in.nextToken();
            return;
        }
        throw error("bytes");
    }

    private void checkFixed(int size) throws IOException {
        advance(Symbol.FIXED);
        Symbol.IntCheckAction top = (Symbol.IntCheckAction) this.parser.popSymbol();
        if (size != top.size) {
            throw new AvroTypeException("Incorrect length for fixed binary: expected " + top.size + " but received " + size + " bytes.");
        }
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public void readFixed(byte[] bytes, int start, int len) throws IOException {
        checkFixed(len);
        if (this.in.getCurrentToken() == JsonToken.VALUE_STRING) {
            byte[] result = readByteArray();
            this.in.nextToken();
            if (result.length != len) {
                throw new AvroTypeException("Expected fixed length " + len + ", but got" + result.length);
            }
            System.arraycopy(result, 0, bytes, start, len);
            return;
        }
        throw error("fixed");
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public void skipFixed(int length) throws IOException {
        checkFixed(length);
        doSkipFixed(length);
    }

    private void doSkipFixed(int length) throws IOException {
        if (this.in.getCurrentToken() == JsonToken.VALUE_STRING) {
            byte[] result = readByteArray();
            this.in.nextToken();
            if (result.length != length) {
                throw new AvroTypeException("Expected fixed length " + length + ", but got" + result.length);
            }
            return;
        }
        throw error("fixed");
    }

    @Override // com.flurry.org.apache.avro.io.ParsingDecoder
    protected void skipFixed() throws IOException {
        advance(Symbol.FIXED);
        Symbol.IntCheckAction top = (Symbol.IntCheckAction) this.parser.popSymbol();
        doSkipFixed(top.size);
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public int readEnum() throws IOException {
        advance(Symbol.ENUM);
        Symbol.EnumLabelsAction top = (Symbol.EnumLabelsAction) this.parser.popSymbol();
        if (this.in.getCurrentToken() == JsonToken.VALUE_STRING) {
            this.in.getText();
            int n = top.findLabel(this.in.getText());
            if (n >= 0) {
                this.in.nextToken();
                return n;
            }
            throw new AvroTypeException("Unknown symbol in enum " + this.in.getText());
        }
        throw error("fixed");
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public long readArrayStart() throws IOException {
        advance(Symbol.ARRAY_START);
        if (this.in.getCurrentToken() == JsonToken.START_ARRAY) {
            this.in.nextToken();
            return doArrayNext();
        }
        throw error("array-start");
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public long arrayNext() throws IOException {
        advance(Symbol.ITEM_END);
        return doArrayNext();
    }

    private long doArrayNext() throws IOException {
        if (this.in.getCurrentToken() == JsonToken.END_ARRAY) {
            this.parser.advance(Symbol.ARRAY_END);
            this.in.nextToken();
            return 0L;
        }
        return 1L;
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public long skipArray() throws IOException {
        advance(Symbol.ARRAY_START);
        if (this.in.getCurrentToken() == JsonToken.START_ARRAY) {
            this.in.skipChildren();
            this.in.nextToken();
            advance(Symbol.ARRAY_END);
            return 0L;
        }
        throw error("array-start");
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public long readMapStart() throws IOException {
        advance(Symbol.MAP_START);
        if (this.in.getCurrentToken() == JsonToken.START_OBJECT) {
            this.in.nextToken();
            return doMapNext();
        }
        throw error("map-start");
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public long mapNext() throws IOException {
        advance(Symbol.ITEM_END);
        return doMapNext();
    }

    private long doMapNext() throws IOException {
        if (this.in.getCurrentToken() == JsonToken.END_OBJECT) {
            this.in.nextToken();
            advance(Symbol.MAP_END);
            return 0L;
        }
        return 1L;
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public long skipMap() throws IOException {
        advance(Symbol.MAP_START);
        if (this.in.getCurrentToken() == JsonToken.START_OBJECT) {
            this.in.skipChildren();
            this.in.nextToken();
            advance(Symbol.MAP_END);
            return 0L;
        }
        throw error("map-start");
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public int readIndex() throws IOException {
        String label;
        advance(Symbol.UNION);
        Symbol.Alternative a = (Symbol.Alternative) this.parser.popSymbol();
        if (this.in.getCurrentToken() == JsonToken.VALUE_NULL) {
            label = DataFileConstants.NULL_CODEC;
        } else if (this.in.getCurrentToken() == JsonToken.START_OBJECT && this.in.nextToken() == JsonToken.FIELD_NAME) {
            label = this.in.getText();
            this.in.nextToken();
            this.parser.pushSymbol(Symbol.UNION_END);
        } else {
            throw error("start-union");
        }
        int n = a.findLabel(label);
        if (n < 0) {
            throw new AvroTypeException("Unknown union branch " + label);
        }
        this.parser.pushSymbol(a.getSymbol(n));
        return n;
    }

    @Override // com.flurry.org.apache.avro.io.parsing.Parser.ActionHandler
    public Symbol doAction(Symbol input, Symbol top) throws IOException {
        List<JsonElement> node;
        if (top instanceof Symbol.FieldAdjustAction) {
            Symbol.FieldAdjustAction fa = (Symbol.FieldAdjustAction) top;
            String name = fa.fname;
            if (this.currentReorderBuffer != null && (node = this.currentReorderBuffer.savedFields.get(name)) != null) {
                this.currentReorderBuffer.savedFields.remove(name);
                this.currentReorderBuffer.origParser = this.in;
                this.in = makeParser(node);
            } else if (this.in.getCurrentToken() == JsonToken.FIELD_NAME) {
                do {
                    String fn = this.in.getText();
                    this.in.nextToken();
                    if (!name.equals(fn)) {
                        if (this.currentReorderBuffer == null) {
                            this.currentReorderBuffer = new ReorderBuffer();
                        }
                        this.currentReorderBuffer.savedFields.put(fn, getVaueAsTree(this.in));
                    }
                } while (this.in.getCurrentToken() == JsonToken.FIELD_NAME);
                throw new AvroTypeException("Expected field name not found: " + fa.fname);
            }
        } else if (top == Symbol.FIELD_END) {
            if (this.currentReorderBuffer != null && this.currentReorderBuffer.origParser != null) {
                this.in = this.currentReorderBuffer.origParser;
                this.currentReorderBuffer.origParser = null;
            }
        } else if (top == Symbol.RECORD_START) {
            if (this.in.getCurrentToken() == JsonToken.START_OBJECT) {
                this.in.nextToken();
                this.reorderBuffers.push(this.currentReorderBuffer);
                this.currentReorderBuffer = null;
            } else {
                throw error("record-start");
            }
        } else if (top == Symbol.RECORD_END || top == Symbol.UNION_END) {
            if (this.in.getCurrentToken() != JsonToken.END_OBJECT) {
                throw error(top == Symbol.RECORD_END ? "record-end" : "union-end");
            }
            this.in.nextToken();
            if (top == Symbol.RECORD_END) {
                if (this.currentReorderBuffer != null && !this.currentReorderBuffer.savedFields.isEmpty()) {
                    throw error("Unknown fields: " + this.currentReorderBuffer.savedFields.keySet());
                }
                this.currentReorderBuffer = this.reorderBuffers.pop();
            }
        } else {
            throw new AvroTypeException("Unknown action symbol " + top);
        }
        return null;
    }

    /* JADX INFO: Access modifiers changed from: private */
    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public static class JsonElement {
        public final JsonToken token;
        public final String value;

        public JsonElement(JsonToken t, String value) {
            this.token = t;
            this.value = value;
        }

        public JsonElement(JsonToken t) {
            this(t, null);
        }
    }

    private static List<JsonElement> getVaueAsTree(JsonParser in) throws IOException {
        int level = 0;
        List<JsonElement> result = new ArrayList<>();
        do {
            JsonToken t = in.getCurrentToken();
            switch (AnonymousClass2.$SwitchMap$org$codehaus$jackson$JsonToken[t.ordinal()]) {
                case 1:
                case 2:
                    level++;
                    result.add(new JsonElement(t));
                    break;
                case 3:
                case 4:
                    level--;
                    result.add(new JsonElement(t));
                    break;
                case 5:
                case 6:
                case IabHelper.BILLING_RESPONSE_RESULT_ITEM_ALREADY_OWNED /* 7 */:
                case 8:
                case 9:
                case TapjoyVideoObject.BUTTON_MAX /* 10 */:
                case 11:
                    result.add(new JsonElement(t, in.getText()));
                    break;
            }
            in.nextToken();
        } while (level != 0);
        result.add(new JsonElement(null));
        return result;
    }

    /* JADX INFO: Access modifiers changed from: package-private */
    /* renamed from: com.flurry.org.apache.avro.io.JsonDecoder$2  reason: invalid class name */
    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public static /* synthetic */ class AnonymousClass2 {
        static final /* synthetic */ int[] $SwitchMap$org$codehaus$jackson$JsonToken = new int[JsonToken.values().length];

        static {
            try {
                $SwitchMap$org$codehaus$jackson$JsonToken[JsonToken.START_OBJECT.ordinal()] = 1;
            } catch (NoSuchFieldError e) {
            }
            try {
                $SwitchMap$org$codehaus$jackson$JsonToken[JsonToken.START_ARRAY.ordinal()] = 2;
            } catch (NoSuchFieldError e2) {
            }
            try {
                $SwitchMap$org$codehaus$jackson$JsonToken[JsonToken.END_OBJECT.ordinal()] = 3;
            } catch (NoSuchFieldError e3) {
            }
            try {
                $SwitchMap$org$codehaus$jackson$JsonToken[JsonToken.END_ARRAY.ordinal()] = 4;
            } catch (NoSuchFieldError e4) {
            }
            try {
                $SwitchMap$org$codehaus$jackson$JsonToken[JsonToken.FIELD_NAME.ordinal()] = 5;
            } catch (NoSuchFieldError e5) {
            }
            try {
                $SwitchMap$org$codehaus$jackson$JsonToken[JsonToken.VALUE_STRING.ordinal()] = 6;
            } catch (NoSuchFieldError e6) {
            }
            try {
                $SwitchMap$org$codehaus$jackson$JsonToken[JsonToken.VALUE_NUMBER_INT.ordinal()] = 7;
            } catch (NoSuchFieldError e7) {
            }
            try {
                $SwitchMap$org$codehaus$jackson$JsonToken[JsonToken.VALUE_NUMBER_FLOAT.ordinal()] = 8;
            } catch (NoSuchFieldError e8) {
            }
            try {
                $SwitchMap$org$codehaus$jackson$JsonToken[JsonToken.VALUE_TRUE.ordinal()] = 9;
            } catch (NoSuchFieldError e9) {
            }
            try {
                $SwitchMap$org$codehaus$jackson$JsonToken[JsonToken.VALUE_FALSE.ordinal()] = 10;
            } catch (NoSuchFieldError e10) {
            }
            try {
                $SwitchMap$org$codehaus$jackson$JsonToken[JsonToken.VALUE_NULL.ordinal()] = 11;
            } catch (NoSuchFieldError e11) {
            }
        }
    }

    private JsonParser makeParser(final List<JsonElement> elements) throws IOException {
        return new JsonParser() { // from class: com.flurry.org.apache.avro.io.JsonDecoder.1
            int pos = 0;

            @Override // com.flurry.org.codehaus.jackson.JsonParser
            public ObjectCodec getCodec() {
                throw new UnsupportedOperationException();
            }

            @Override // com.flurry.org.codehaus.jackson.JsonParser
            public void setCodec(ObjectCodec c) {
                throw new UnsupportedOperationException();
            }

            @Override // com.flurry.org.codehaus.jackson.JsonParser, java.io.Closeable, java.lang.AutoCloseable
            public void close() throws IOException {
                throw new UnsupportedOperationException();
            }

            @Override // com.flurry.org.codehaus.jackson.JsonParser
            public JsonToken nextToken() throws IOException {
                this.pos++;
                return ((JsonElement) elements.get(this.pos)).token;
            }

            @Override // com.flurry.org.codehaus.jackson.JsonParser
            public JsonParser skipChildren() throws IOException {
                int level = 0;
                do {
                    int[] iArr = AnonymousClass2.$SwitchMap$org$codehaus$jackson$JsonToken;
                    List list = elements;
                    int i = this.pos;
                    this.pos = i + 1;
                    switch (iArr[((JsonElement) list.get(i)).token.ordinal()]) {
                        case 1:
                        case 2:
                            level++;
                            continue;
                        case 3:
                        case 4:
                            level--;
                            continue;
                    }
                } while (level > 0);
                return this;
            }

            @Override // com.flurry.org.codehaus.jackson.JsonParser
            public boolean isClosed() {
                throw new UnsupportedOperationException();
            }

            @Override // com.flurry.org.codehaus.jackson.JsonParser
            public String getCurrentName() throws IOException {
                throw new UnsupportedOperationException();
            }

            @Override // com.flurry.org.codehaus.jackson.JsonParser
            public JsonStreamContext getParsingContext() {
                throw new UnsupportedOperationException();
            }

            @Override // com.flurry.org.codehaus.jackson.JsonParser
            public JsonLocation getTokenLocation() {
                throw new UnsupportedOperationException();
            }

            @Override // com.flurry.org.codehaus.jackson.JsonParser
            public JsonLocation getCurrentLocation() {
                throw new UnsupportedOperationException();
            }

            @Override // com.flurry.org.codehaus.jackson.JsonParser
            public String getText() throws IOException {
                return ((JsonElement) elements.get(this.pos)).value;
            }

            @Override // com.flurry.org.codehaus.jackson.JsonParser
            public char[] getTextCharacters() throws IOException {
                throw new UnsupportedOperationException();
            }

            @Override // com.flurry.org.codehaus.jackson.JsonParser
            public int getTextLength() throws IOException {
                throw new UnsupportedOperationException();
            }

            @Override // com.flurry.org.codehaus.jackson.JsonParser
            public int getTextOffset() throws IOException {
                throw new UnsupportedOperationException();
            }

            @Override // com.flurry.org.codehaus.jackson.JsonParser
            public Number getNumberValue() throws IOException {
                throw new UnsupportedOperationException();
            }

            @Override // com.flurry.org.codehaus.jackson.JsonParser
            public JsonParser.NumberType getNumberType() throws IOException {
                throw new UnsupportedOperationException();
            }

            @Override // com.flurry.org.codehaus.jackson.JsonParser
            public int getIntValue() throws IOException {
                return Integer.parseInt(getText());
            }

            @Override // com.flurry.org.codehaus.jackson.JsonParser
            public long getLongValue() throws IOException {
                return Long.parseLong(getText());
            }

            @Override // com.flurry.org.codehaus.jackson.JsonParser
            public BigInteger getBigIntegerValue() throws IOException {
                throw new UnsupportedOperationException();
            }

            @Override // com.flurry.org.codehaus.jackson.JsonParser
            public float getFloatValue() throws IOException {
                return Float.parseFloat(getText());
            }

            @Override // com.flurry.org.codehaus.jackson.JsonParser
            public double getDoubleValue() throws IOException {
                return Double.parseDouble(getText());
            }

            @Override // com.flurry.org.codehaus.jackson.JsonParser
            public BigDecimal getDecimalValue() throws IOException {
                throw new UnsupportedOperationException();
            }

            @Override // com.flurry.org.codehaus.jackson.JsonParser
            public byte[] getBinaryValue(Base64Variant b64variant) throws IOException {
                throw new UnsupportedOperationException();
            }

            @Override // com.flurry.org.codehaus.jackson.JsonParser
            public JsonToken getCurrentToken() {
                return ((JsonElement) elements.get(this.pos)).token;
            }
        };
    }

    private AvroTypeException error(String type) {
        return new AvroTypeException("Expected " + type + ". Got " + this.in.getCurrentToken());
    }
}
