package com.flurry.org.apache.avro.io;

import com.flurry.android.Constants;
import com.flurry.org.apache.avro.util.ByteBufferInputStream;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;

/* JADX INFO: Access modifiers changed from: package-private */
/* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
public class DirectBinaryDecoder extends BinaryDecoder {
    private final byte[] buf = new byte[8];
    private ByteReader byteReader;
    private InputStream in;

    /* JADX INFO: Access modifiers changed from: private */
    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public class ByteReader {
        private ByteReader() {
        }

        public ByteBuffer read(ByteBuffer old, int length) throws IOException {
            ByteBuffer result;
            if (old != null && length <= old.capacity()) {
                result = old;
                result.clear();
            } else {
                result = ByteBuffer.allocate(length);
            }
            DirectBinaryDecoder.this.doReadBytes(result.array(), result.position(), length);
            result.limit(length);
            return result;
        }
    }

    /* JADX INFO: Access modifiers changed from: private */
    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public class ReuseByteReader extends ByteReader {
        private final ByteBufferInputStream bbi;

        public ReuseByteReader(ByteBufferInputStream bbi) {
            super();
            this.bbi = bbi;
        }

        @Override // com.flurry.org.apache.avro.io.DirectBinaryDecoder.ByteReader
        public ByteBuffer read(ByteBuffer old, int length) throws IOException {
            return old != null ? super.read(old, length) : this.bbi.readBuffer(length);
        }
    }

    /* JADX INFO: Access modifiers changed from: package-private */
    public DirectBinaryDecoder(InputStream in) {
        configure(in);
    }

    /* JADX INFO: Access modifiers changed from: package-private */
    public DirectBinaryDecoder configure(InputStream in) {
        this.in = in;
        this.byteReader = in instanceof ByteBufferInputStream ? new ReuseByteReader((ByteBufferInputStream) in) : new ByteReader();
        return this;
    }

    @Override // com.flurry.org.apache.avro.io.BinaryDecoder, com.flurry.org.apache.avro.io.Decoder
    public boolean readBoolean() throws IOException {
        int n = this.in.read();
        if (n < 0) {
            throw new EOFException();
        }
        return n == 1;
    }

    @Override // com.flurry.org.apache.avro.io.BinaryDecoder, com.flurry.org.apache.avro.io.Decoder
    public int readInt() throws IOException {
        int n = 0;
        int shift = 0;
        do {
            int b = this.in.read();
            if (b >= 0) {
                n |= (b & 127) << shift;
                if ((b & 128) == 0) {
                    return (n >>> 1) ^ (-(n & 1));
                }
                shift += 7;
            } else {
                throw new EOFException();
            }
        } while (shift < 32);
        throw new IOException("Invalid int encoding");
    }

    @Override // com.flurry.org.apache.avro.io.BinaryDecoder, com.flurry.org.apache.avro.io.Decoder
    public long readLong() throws IOException {
        long n = 0;
        int shift = 0;
        do {
            int b = this.in.read();
            if (b >= 0) {
                n |= (b & 127) << shift;
                if ((b & 128) == 0) {
                    return (n >>> 1) ^ (-(1 & n));
                }
                shift += 7;
            } else {
                throw new EOFException();
            }
        } while (shift < 64);
        throw new IOException("Invalid long encoding");
    }

    @Override // com.flurry.org.apache.avro.io.BinaryDecoder, com.flurry.org.apache.avro.io.Decoder
    public float readFloat() throws IOException {
        doReadBytes(this.buf, 0, 4);
        int n = (this.buf[0] & Constants.UNKNOWN) | ((this.buf[1] & Constants.UNKNOWN) << 8) | ((this.buf[2] & Constants.UNKNOWN) << 16) | ((this.buf[3] & Constants.UNKNOWN) << 24);
        return Float.intBitsToFloat(n);
    }

    @Override // com.flurry.org.apache.avro.io.BinaryDecoder, com.flurry.org.apache.avro.io.Decoder
    public double readDouble() throws IOException {
        doReadBytes(this.buf, 0, 8);
        long n = (this.buf[0] & 255) | ((this.buf[1] & 255) << 8) | ((this.buf[2] & 255) << 16) | ((this.buf[3] & 255) << 24) | ((this.buf[4] & 255) << 32) | ((this.buf[5] & 255) << 40) | ((this.buf[6] & 255) << 48) | ((this.buf[7] & 255) << 56);
        return Double.longBitsToDouble(n);
    }

    @Override // com.flurry.org.apache.avro.io.BinaryDecoder, com.flurry.org.apache.avro.io.Decoder
    public ByteBuffer readBytes(ByteBuffer old) throws IOException {
        int length = readInt();
        return this.byteReader.read(old, length);
    }

    @Override // com.flurry.org.apache.avro.io.BinaryDecoder
    protected void doSkipBytes(long length) throws IOException {
        while (length > 0) {
            long n = this.in.skip(length);
            if (n <= 0) {
                throw new EOFException();
            }
            length -= n;
        }
    }

    @Override // com.flurry.org.apache.avro.io.BinaryDecoder
    protected void doReadBytes(byte[] bytes, int start, int length) throws IOException {
        while (true) {
            int n = this.in.read(bytes, start, length);
            if (n == length || length == 0) {
                return;
            }
            if (n < 0) {
                throw new EOFException();
            }
            start += n;
            length -= n;
        }
    }

    @Override // com.flurry.org.apache.avro.io.BinaryDecoder
    public InputStream inputStream() {
        return this.in;
    }

    @Override // com.flurry.org.apache.avro.io.BinaryDecoder
    public boolean isEnd() throws IOException {
        throw new UnsupportedOperationException();
    }
}
