package com.flurry.org.apache.avro.io;

import com.flurry.org.apache.avro.AvroRuntimeException;
import java.io.IOException;
import java.io.OutputStream;

/* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
public class BufferedBinaryEncoder extends BinaryEncoder {
    private byte[] buf;
    private int bulkLimit;
    private int pos;
    private ByteSink sink;

    /* JADX INFO: Access modifiers changed from: package-private */
    public BufferedBinaryEncoder(OutputStream out, int bufferSize) {
        configure(out, bufferSize);
    }

    /* JADX INFO: Access modifiers changed from: package-private */
    public BufferedBinaryEncoder configure(OutputStream out, int bufferSize) {
        if (out == null) {
            throw new NullPointerException("OutputStream cannot be null!");
        }
        if (this.sink != null && this.pos > 0) {
            try {
                flushBuffer();
            } catch (IOException e) {
                throw new AvroRuntimeException("Failure flushing old output", e);
            }
        }
        this.sink = new OutputStreamSink(out);
        this.pos = 0;
        if (this.buf == null || this.buf.length != bufferSize) {
            this.buf = new byte[bufferSize];
        }
        this.bulkLimit = this.buf.length >>> 1;
        if (this.bulkLimit > 512) {
            this.bulkLimit = 512;
        }
        return this;
    }

    @Override // java.io.Flushable
    public void flush() throws IOException {
        flushBuffer();
        this.sink.innerFlush();
    }

    private void flushBuffer() throws IOException {
        if (this.pos > 0) {
            this.sink.innerWrite(this.buf, 0, this.pos);
            this.pos = 0;
        }
    }

    private void ensureBounds(int num) throws IOException {
        int remaining = this.buf.length - this.pos;
        if (remaining < num) {
            flushBuffer();
        }
    }

    @Override // com.flurry.org.apache.avro.io.Encoder
    public void writeBoolean(boolean b) throws IOException {
        if (this.buf.length == this.pos) {
            flushBuffer();
        }
        this.pos += BinaryData.encodeBoolean(b, this.buf, this.pos);
    }

    @Override // com.flurry.org.apache.avro.io.Encoder
    public void writeInt(int n) throws IOException {
        ensureBounds(5);
        this.pos += BinaryData.encodeInt(n, this.buf, this.pos);
    }

    @Override // com.flurry.org.apache.avro.io.Encoder
    public void writeLong(long n) throws IOException {
        ensureBounds(10);
        this.pos += BinaryData.encodeLong(n, this.buf, this.pos);
    }

    @Override // com.flurry.org.apache.avro.io.Encoder
    public void writeFloat(float f) throws IOException {
        ensureBounds(4);
        this.pos += BinaryData.encodeFloat(f, this.buf, this.pos);
    }

    @Override // com.flurry.org.apache.avro.io.Encoder
    public void writeDouble(double d) throws IOException {
        ensureBounds(8);
        this.pos += BinaryData.encodeDouble(d, this.buf, this.pos);
    }

    @Override // com.flurry.org.apache.avro.io.Encoder
    public void writeFixed(byte[] bytes, int start, int len) throws IOException {
        if (len > this.bulkLimit) {
            flushBuffer();
            this.sink.innerWrite(bytes, start, len);
            return;
        }
        ensureBounds(len);
        System.arraycopy(bytes, start, this.buf, this.pos, len);
        this.pos += len;
    }

    @Override // com.flurry.org.apache.avro.io.BinaryEncoder
    protected void writeZero() throws IOException {
        writeByte(0);
    }

    private void writeByte(int b) throws IOException {
        if (this.pos == this.buf.length) {
            flushBuffer();
        }
        byte[] bArr = this.buf;
        int i = this.pos;
        this.pos = i + 1;
        bArr[i] = (byte) (b & 255);
    }

    @Override // com.flurry.org.apache.avro.io.BinaryEncoder
    public int bytesBuffered() {
        return this.pos;
    }

    /* JADX INFO: Access modifiers changed from: private */
    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public static abstract class ByteSink {
        protected abstract void innerFlush() throws IOException;

        protected abstract void innerWrite(byte[] bArr, int i, int i2) throws IOException;

        protected ByteSink() {
        }
    }

    /* JADX INFO: Access modifiers changed from: package-private */
    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public static class OutputStreamSink extends ByteSink {
        private final OutputStream out;

        private OutputStreamSink(OutputStream out) {
            this.out = out;
        }

        @Override // com.flurry.org.apache.avro.io.BufferedBinaryEncoder.ByteSink
        protected void innerWrite(byte[] bytes, int off, int len) throws IOException {
            this.out.write(bytes, off, len);
        }

        @Override // com.flurry.org.apache.avro.io.BufferedBinaryEncoder.ByteSink
        protected void innerFlush() throws IOException {
            this.out.flush();
        }
    }
}
