package com.flurry.org.apache.avro.io;

import com.flurry.android.Constants;
import com.flurry.org.apache.avro.util.ByteBufferOutputStream;
import com.flurry.org.apache.avro.util.Utf8;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;

/* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
public class BinaryDecoder extends Decoder {
    private ByteSource source = null;
    private byte[] buf = null;
    private int minPos = 0;
    private int pos = 0;
    private int limit = 0;
    private final Utf8 scratchUtf8 = new Utf8();

    /* JADX INFO: Access modifiers changed from: package-private */
    public BufferAccessor getBufferAccessor() {
        return new BufferAccessor();
    }

    /* JADX INFO: Access modifiers changed from: protected */
    public BinaryDecoder() {
    }

    /* JADX INFO: Access modifiers changed from: package-private */
    public BinaryDecoder(InputStream in, int bufferSize) {
        configure(in, bufferSize);
    }

    /* JADX INFO: Access modifiers changed from: package-private */
    public BinaryDecoder(byte[] data, int offset, int length) {
        configure(data, offset, length);
    }

    /* JADX INFO: Access modifiers changed from: package-private */
    public BinaryDecoder configure(InputStream in, int bufferSize) {
        configureSource(bufferSize, new InputStreamByteSource(in));
        return this;
    }

    /* JADX INFO: Access modifiers changed from: package-private */
    public BinaryDecoder configure(byte[] data, int offset, int length) {
        configureSource(ByteBufferOutputStream.BUFFER_SIZE, new ByteArrayByteSource(data, offset, length));
        return this;
    }

    private void configureSource(int bufferSize, ByteSource source) {
        if (this.source != null) {
            this.source.detach();
        }
        source.attach(bufferSize, this);
        this.source = source;
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public void readNull() throws IOException {
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public boolean readBoolean() throws IOException {
        if (this.limit == this.pos) {
            this.limit = this.source.tryReadRaw(this.buf, 0, this.buf.length);
            this.pos = 0;
            if (this.limit == 0) {
                throw new EOFException();
            }
        }
        byte[] bArr = this.buf;
        int i = this.pos;
        this.pos = i + 1;
        int n = bArr[i] & Constants.UNKNOWN;
        return n == 1;
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public int readInt() throws IOException {
        ensureBounds(5);
        int len = 1;
        int b = this.buf[this.pos] & Constants.UNKNOWN;
        int n = b & 127;
        if (b > 127) {
            int len2 = 1 + 1;
            int b2 = this.buf[this.pos + 1] & Constants.UNKNOWN;
            n ^= (b2 & 127) << 7;
            if (b2 > 127) {
                len = len2 + 1;
                int b3 = this.buf[this.pos + 2] & Constants.UNKNOWN;
                n ^= (b3 & 127) << 14;
                if (b3 > 127) {
                    len++;
                    int b4 = this.buf[this.pos + 3] & Constants.UNKNOWN;
                    n ^= (b4 & 127) << 21;
                    if (b4 > 127) {
                        len++;
                        int b5 = this.buf[this.pos + 4] & Constants.UNKNOWN;
                        n ^= (b5 & 127) << 28;
                        if (b5 > 127) {
                            throw new IOException("Invalid int encoding");
                        }
                    }
                }
            } else {
                len = len2;
            }
        }
        this.pos += len;
        if (this.pos > this.limit) {
            throw new EOFException();
        }
        return (n >>> 1) ^ (-(n & 1));
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public long readLong() throws IOException {
        long l;
        ensureBounds(10);
        byte[] bArr = this.buf;
        int i = this.pos;
        this.pos = i + 1;
        int b = bArr[i] & Constants.UNKNOWN;
        int n = b & 127;
        if (b > 127) {
            byte[] bArr2 = this.buf;
            int i2 = this.pos;
            this.pos = i2 + 1;
            int b2 = bArr2[i2] & Constants.UNKNOWN;
            int n2 = n ^ ((b2 & 127) << 7);
            if (b2 > 127) {
                byte[] bArr3 = this.buf;
                int i3 = this.pos;
                this.pos = i3 + 1;
                int b3 = bArr3[i3] & Constants.UNKNOWN;
                int n3 = n2 ^ ((b3 & 127) << 14);
                if (b3 > 127) {
                    byte[] bArr4 = this.buf;
                    int i4 = this.pos;
                    this.pos = i4 + 1;
                    int b4 = bArr4[i4] & Constants.UNKNOWN;
                    int n4 = n3 ^ ((b4 & 127) << 21);
                    if (b4 > 127) {
                        l = innerLongDecode(n4);
                    } else {
                        l = n4;
                    }
                } else {
                    l = n3;
                }
            } else {
                l = n2;
            }
        } else {
            l = n;
        }
        if (this.pos > this.limit) {
            throw new EOFException();
        }
        return (l >>> 1) ^ (-(1 & l));
    }

    private long innerLongDecode(long l) throws IOException {
        int len = 1;
        int b = this.buf[this.pos] & Constants.UNKNOWN;
        long l2 = l ^ ((b & 127) << 28);
        if (b > 127) {
            int len2 = 1 + 1;
            int b2 = this.buf[this.pos + 1] & Constants.UNKNOWN;
            l2 ^= (b2 & 127) << 35;
            if (b2 > 127) {
                len = len2 + 1;
                int b3 = this.buf[this.pos + 2] & Constants.UNKNOWN;
                l2 ^= (b3 & 127) << 42;
                if (b3 > 127) {
                    len++;
                    int b4 = this.buf[this.pos + 3] & Constants.UNKNOWN;
                    l2 ^= (b4 & 127) << 49;
                    if (b4 > 127) {
                        len++;
                        int b5 = this.buf[this.pos + 4] & Constants.UNKNOWN;
                        l2 ^= (b5 & 127) << 56;
                        if (b5 > 127) {
                            len++;
                            int b6 = this.buf[this.pos + 5] & Constants.UNKNOWN;
                            l2 ^= (b6 & 127) << 63;
                            if (b6 > 127) {
                                throw new IOException("Invalid long encoding");
                            }
                        }
                    }
                }
            } else {
                len = len2;
            }
        }
        this.pos += len;
        return l2;
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public float readFloat() throws IOException {
        ensureBounds(4);
        int len = 1 + 1;
        int i = len + 1 + 1;
        int n = (this.buf[this.pos] & Constants.UNKNOWN) | ((this.buf[this.pos + 1] & Constants.UNKNOWN) << 8) | ((this.buf[this.pos + 2] & Constants.UNKNOWN) << 16) | ((this.buf[this.pos + 3] & Constants.UNKNOWN) << 24);
        if (this.pos + 4 > this.limit) {
            throw new EOFException();
        }
        this.pos += 4;
        return Float.intBitsToFloat(n);
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public double readDouble() throws IOException {
        ensureBounds(8);
        int len = 1 + 1;
        int n1 = (this.buf[this.pos] & Constants.UNKNOWN) | ((this.buf[this.pos + 1] & Constants.UNKNOWN) << 8) | ((this.buf[this.pos + 2] & Constants.UNKNOWN) << 16) | ((this.buf[this.pos + 3] & Constants.UNKNOWN) << 24);
        int i = len + 1 + 1 + 1 + 1 + 1 + 1;
        int n2 = (this.buf[this.pos + 4] & Constants.UNKNOWN) | ((this.buf[this.pos + 5] & Constants.UNKNOWN) << 8) | ((this.buf[this.pos + 6] & Constants.UNKNOWN) << 16) | ((this.buf[this.pos + 7] & Constants.UNKNOWN) << 24);
        if (this.pos + 8 > this.limit) {
            throw new EOFException();
        }
        this.pos += 8;
        return Double.longBitsToDouble((n1 & 4294967295L) | (n2 << 32));
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public Utf8 readString(Utf8 old) throws IOException {
        int length = readInt();
        Utf8 result = old != null ? old : new Utf8();
        result.setByteLength(length);
        if (length != 0) {
            doReadBytes(result.getBytes(), 0, length);
        }
        return result;
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public String readString() throws IOException {
        return readString(this.scratchUtf8).toString();
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public void skipString() throws IOException {
        doSkipBytes(readInt());
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public ByteBuffer readBytes(ByteBuffer old) throws IOException {
        ByteBuffer result;
        int length = readInt();
        if (old != null && length <= old.capacity()) {
            result = old;
            result.clear();
        } else {
            result = ByteBuffer.allocate(length);
        }
        doReadBytes(result.array(), result.position(), length);
        result.limit(length);
        return result;
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public void skipBytes() throws IOException {
        doSkipBytes(readInt());
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public void readFixed(byte[] bytes, int start, int length) throws IOException {
        doReadBytes(bytes, start, length);
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public void skipFixed(int length) throws IOException {
        doSkipBytes(length);
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public int readEnum() throws IOException {
        return readInt();
    }

    protected void doSkipBytes(long length) throws IOException {
        int remaining = this.limit - this.pos;
        if (length <= remaining) {
            this.pos = (int) (this.pos + length);
            return;
        }
        this.pos = 0;
        this.limit = 0;
        this.source.skipSourceBytes(length - remaining);
    }

    protected void doReadBytes(byte[] bytes, int start, int length) throws IOException {
        int remaining = this.limit - this.pos;
        if (length <= remaining) {
            System.arraycopy(this.buf, this.pos, bytes, start, length);
            this.pos += length;
            return;
        }
        System.arraycopy(this.buf, this.pos, bytes, start, remaining);
        this.pos = this.limit;
        this.source.readRaw(bytes, start + remaining, length - remaining);
    }

    protected long doReadItemCount() throws IOException {
        long result = readLong();
        if (result < 0) {
            readLong();
            return -result;
        }
        return result;
    }

    private long doSkipItems() throws IOException {
        int readInt = readInt();
        while (true) {
            long result = readInt;
            if (result < 0) {
                long bytecount = readLong();
                doSkipBytes(bytecount);
                readInt = readInt();
            } else {
                return result;
            }
        }
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public long readArrayStart() throws IOException {
        return doReadItemCount();
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public long arrayNext() throws IOException {
        return doReadItemCount();
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public long skipArray() throws IOException {
        return doSkipItems();
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public long readMapStart() throws IOException {
        return doReadItemCount();
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public long mapNext() throws IOException {
        return doReadItemCount();
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public long skipMap() throws IOException {
        return doSkipItems();
    }

    @Override // com.flurry.org.apache.avro.io.Decoder
    public int readIndex() throws IOException {
        return readInt();
    }

    public boolean isEnd() throws IOException {
        if (this.limit - this.pos > 0) {
            return false;
        }
        if (this.source.isEof()) {
            return true;
        }
        int read = this.source.tryReadRaw(this.buf, 0, this.buf.length);
        this.pos = 0;
        this.limit = read;
        return read == 0;
    }

    private void ensureBounds(int num) throws IOException {
        int remaining = this.limit - this.pos;
        if (remaining < num) {
            this.source.compactAndFill(this.buf, this.pos, this.minPos, remaining);
        }
    }

    public InputStream inputStream() {
        return this.source;
    }

    /* JADX INFO: Access modifiers changed from: package-private */
    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public static class BufferAccessor {
        private byte[] buf;
        private final BinaryDecoder decoder;
        boolean detached;
        private int limit;
        private int pos;

        private BufferAccessor(BinaryDecoder decoder) {
            this.detached = false;
            this.decoder = decoder;
        }

        void detach() {
            this.buf = this.decoder.buf;
            this.pos = this.decoder.pos;
            this.limit = this.decoder.limit;
            this.detached = true;
        }

        /* JADX INFO: Access modifiers changed from: package-private */
        public int getPos() {
            if (!this.detached) {
                return this.decoder.pos;
            }
            return this.pos;
        }

        int getLim() {
            if (!this.detached) {
                return this.decoder.limit;
            }
            return this.limit;
        }

        /* JADX INFO: Access modifiers changed from: package-private */
        public byte[] getBuf() {
            if (!this.detached) {
                return this.decoder.buf;
            }
            return this.buf;
        }

        void setPos(int pos) {
            if (!this.detached) {
                this.decoder.pos = pos;
            } else {
                this.pos = pos;
            }
        }

        void setLimit(int limit) {
            if (!this.detached) {
                this.decoder.limit = limit;
            } else {
                this.limit = limit;
            }
        }

        void setBuf(byte[] buf, int offset, int length) {
            if (!this.detached) {
                this.decoder.buf = buf;
                this.decoder.limit = offset + length;
                this.decoder.pos = offset;
                this.decoder.minPos = offset;
                return;
            }
            this.buf = buf;
            this.limit = offset + length;
            this.pos = offset;
        }
    }

    /* JADX INFO: Access modifiers changed from: package-private */
    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public static abstract class ByteSource extends InputStream {
        protected BufferAccessor ba;

        abstract boolean isEof();

        protected abstract void readRaw(byte[] bArr, int i, int i2) throws IOException;

        protected abstract void skipSourceBytes(long j) throws IOException;

        protected abstract int tryReadRaw(byte[] bArr, int i, int i2) throws IOException;

        protected abstract long trySkipBytes(long j) throws IOException;

        protected ByteSource() {
        }

        protected void attach(int bufferSize, BinaryDecoder decoder) {
            decoder.buf = new byte[bufferSize];
            decoder.pos = 0;
            decoder.minPos = 0;
            decoder.limit = 0;
            this.ba = new BufferAccessor();
        }

        protected void detach() {
            this.ba.detach();
        }

        protected void compactAndFill(byte[] buf, int pos, int minPos, int remaining) throws IOException {
            System.arraycopy(buf, pos, buf, minPos, remaining);
            this.ba.setPos(minPos);
            int newLimit = remaining + tryReadRaw(buf, minPos + remaining, buf.length - remaining);
            this.ba.setLimit(newLimit);
        }

        @Override // java.io.InputStream
        public int read(byte[] b, int off, int len) throws IOException {
            int lim = this.ba.getLim();
            int pos = this.ba.getPos();
            byte[] buf = this.ba.getBuf();
            int remaining = lim - pos;
            if (remaining >= len) {
                System.arraycopy(buf, pos, b, off, len);
                this.ba.setPos(pos + len);
                return len;
            }
            System.arraycopy(buf, pos, b, off, remaining);
            this.ba.setPos(pos + remaining);
            int inputRead = remaining + tryReadRaw(b, off + remaining, len - remaining);
            if (inputRead == 0) {
                return -1;
            }
            return inputRead;
        }

        @Override // java.io.InputStream
        public long skip(long n) throws IOException {
            int lim = this.ba.getLim();
            int pos = this.ba.getPos();
            int remaining = lim - pos;
            if (remaining > n) {
                this.ba.setPos((int) (pos + n));
                return n;
            }
            this.ba.setPos(lim);
            long isSkipCount = trySkipBytes(n - remaining);
            return isSkipCount + remaining;
        }

        @Override // java.io.InputStream
        public int available() throws IOException {
            return this.ba.getLim() - this.ba.getPos();
        }
    }

    /* JADX INFO: Access modifiers changed from: private */
    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public static class InputStreamByteSource extends ByteSource {
        private InputStream in;
        protected boolean isEof;

        private InputStreamByteSource(InputStream in) {
            this.isEof = false;
            this.in = in;
        }

        @Override // com.flurry.org.apache.avro.io.BinaryDecoder.ByteSource
        protected void skipSourceBytes(long length) throws IOException {
            boolean readZero = false;
            while (length > 0) {
                long n = this.in.skip(length);
                if (n > 0) {
                    length -= n;
                } else if (n == 0) {
                    if (readZero) {
                        this.isEof = true;
                        throw new EOFException();
                    }
                    readZero = true;
                } else {
                    this.isEof = true;
                    throw new EOFException();
                }
            }
        }

        /* JADX WARN: Unsupported multi-entry loop pattern (BACK_EDGE: B:18:0x0028 -> B:12:0x001e). Please submit an issue!!! */
        @Override // com.flurry.org.apache.avro.io.BinaryDecoder.ByteSource
        protected long trySkipBytes(long length) throws IOException {
            long leftToSkip = length;
            boolean readZero = false;
            while (true) {
                if (leftToSkip <= 0) {
                    break;
                }
                try {
                    long n = this.in.skip(length);
                    if (n > 0) {
                        leftToSkip -= n;
                    } else if (n == 0) {
                        if (readZero) {
                            this.isEof = true;
                            break;
                        }
                        readZero = true;
                    } else {
                        this.isEof = true;
                        break;
                    }
                } catch (EOFException e) {
                    this.isEof = true;
                }
            }
            return length - leftToSkip;
        }

        @Override // com.flurry.org.apache.avro.io.BinaryDecoder.ByteSource
        protected void readRaw(byte[] data, int off, int len) throws IOException {
            while (len > 0) {
                int read = this.in.read(data, off, len);
                if (read < 0) {
                    this.isEof = true;
                    throw new EOFException();
                } else {
                    len -= read;
                    off += read;
                }
            }
        }

        /* JADX WARN: Code restructure failed: missing block: B:6:0x000c, code lost:
            r5.isEof = true;
         */
        @Override // com.flurry.org.apache.avro.io.BinaryDecoder.ByteSource
        /*
            Code decompiled incorrectly, please refer to instructions dump.
            To view partially-correct code enable 'Show inconsistent code' option in preferences
        */
        protected int tryReadRaw(byte[] r6, int r7, int r8) throws java.io.IOException {
            /*
                r5 = this;
                r4 = 1
                r1 = r8
            L2:
                if (r1 <= 0) goto Lf
                java.io.InputStream r3 = r5.in     // Catch: java.io.EOFException -> L15
                int r2 = r3.read(r6, r7, r1)     // Catch: java.io.EOFException -> L15
                if (r2 >= 0) goto L12
                r3 = 1
                r5.isEof = r3     // Catch: java.io.EOFException -> L15
            Lf:
                int r3 = r8 - r1
                return r3
            L12:
                int r1 = r1 - r2
                int r7 = r7 + r2
                goto L2
            L15:
                r0 = move-exception
                r5.isEof = r4
                goto Lf
            */
            throw new UnsupportedOperationException("Method not decompiled: com.flurry.org.apache.avro.io.BinaryDecoder.InputStreamByteSource.tryReadRaw(byte[], int, int):int");
        }

        @Override // java.io.InputStream
        public int read() throws IOException {
            if (this.ba.getLim() - this.ba.getPos() == 0) {
                return this.in.read();
            }
            int position = this.ba.getPos();
            int i = this.ba.getBuf()[position] & Constants.UNKNOWN;
            this.ba.setPos(position + 1);
            return i;
        }

        @Override // com.flurry.org.apache.avro.io.BinaryDecoder.ByteSource
        public boolean isEof() {
            return this.isEof;
        }

        @Override // java.io.InputStream, java.io.Closeable, java.lang.AutoCloseable
        public void close() throws IOException {
            this.in.close();
        }
    }

    /* JADX INFO: Access modifiers changed from: private */
    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public static class ByteArrayByteSource extends ByteSource {
        private boolean compacted;
        private byte[] data;
        private int max;
        private int position;

        private ByteArrayByteSource(byte[] data, int start, int len) {
            this.compacted = false;
            if (data.length < 16 || len < 16) {
                this.data = new byte[16];
                System.arraycopy(data, start, this.data, 0, len);
                this.position = 0;
                this.max = len;
                return;
            }
            this.data = data;
            this.position = start;
            this.max = start + len;
        }

        @Override // com.flurry.org.apache.avro.io.BinaryDecoder.ByteSource
        protected void attach(int bufferSize, BinaryDecoder decoder) {
            decoder.buf = this.data;
            decoder.pos = this.position;
            decoder.minPos = this.position;
            decoder.limit = this.max;
            this.ba = new BufferAccessor();
        }

        @Override // com.flurry.org.apache.avro.io.BinaryDecoder.ByteSource
        protected void skipSourceBytes(long length) throws IOException {
            long skipped = trySkipBytes(length);
            if (skipped < length) {
                throw new EOFException();
            }
        }

        @Override // com.flurry.org.apache.avro.io.BinaryDecoder.ByteSource
        protected long trySkipBytes(long length) throws IOException {
            this.max = this.ba.getLim();
            this.position = this.ba.getPos();
            long remaining = this.max - this.position;
            if (remaining >= length) {
                this.position = (int) (this.position + length);
                this.ba.setPos(this.position);
                return length;
            }
            this.position = (int) (this.position + remaining);
            this.ba.setPos(this.position);
            return remaining;
        }

        @Override // com.flurry.org.apache.avro.io.BinaryDecoder.ByteSource
        protected void readRaw(byte[] data, int off, int len) throws IOException {
            int read = tryReadRaw(data, off, len);
            if (read < len) {
                throw new EOFException();
            }
        }

        @Override // com.flurry.org.apache.avro.io.BinaryDecoder.ByteSource
        protected int tryReadRaw(byte[] data, int off, int len) throws IOException {
            return 0;
        }

        @Override // com.flurry.org.apache.avro.io.BinaryDecoder.ByteSource
        protected void compactAndFill(byte[] buf, int pos, int minPos, int remaining) throws IOException {
            if (!this.compacted) {
                byte[] tinybuf = new byte[remaining + 16];
                System.arraycopy(buf, pos, tinybuf, 0, remaining);
                this.ba.setBuf(tinybuf, 0, remaining);
                this.compacted = true;
            }
        }

        @Override // java.io.InputStream
        public int read() throws IOException {
            this.max = this.ba.getLim();
            this.position = this.ba.getPos();
            if (this.position >= this.max) {
                return -1;
            }
            byte[] buf = this.ba.getBuf();
            int i = this.position;
            this.position = i + 1;
            int i2 = buf[i] & Constants.UNKNOWN;
            this.ba.setPos(this.position);
            return i2;
        }

        @Override // java.io.InputStream, java.io.Closeable, java.lang.AutoCloseable
        public void close() throws IOException {
            this.ba.setPos(this.ba.getLim());
        }

        @Override // com.flurry.org.apache.avro.io.BinaryDecoder.ByteSource
        public boolean isEof() {
            int remaining = this.ba.getLim() - this.ba.getPos();
            return remaining == 0;
        }
    }
}
