package com.flurry.org.apache.avro.generic;

import com.flurry.org.apache.avro.AvroRuntimeException;
import com.flurry.org.apache.avro.AvroTypeException;
import com.flurry.org.apache.avro.Schema;
import com.flurry.org.apache.avro.UnresolvedUnionException;
import com.flurry.org.apache.avro.file.DataFileConstants;
import com.flurry.org.apache.avro.io.BinaryData;
import com.flurry.org.apache.avro.io.DatumReader;
import com.flurry.org.apache.avro.util.Utf8;
import com.prime31.util.IabHelper;
import com.tapjoy.TapjoyVideoObject;
import java.nio.ByteBuffer;
import java.util.AbstractList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

/* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
public class GenericData {
    private static final GenericData INSTANCE = new GenericData();
    private static final Schema STRINGS = Schema.create(Schema.Type.STRING);
    protected static final String STRING_PROP = "avro.java.string";
    protected static final String STRING_TYPE_STRING = "String";

    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public enum StringType {
        CharSequence,
        String,
        Utf8
    }

    public static void setStringType(Schema s, StringType stringType) {
        if (stringType == StringType.String) {
            s.addProp(STRING_PROP, STRING_TYPE_STRING);
        }
    }

    public static GenericData get() {
        return INSTANCE;
    }

    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public static class Record implements GenericRecord, Comparable<Record> {
        private final Schema schema;
        private final Object[] values;

        public Record(Schema schema) {
            if (schema == null || !Schema.Type.RECORD.equals(schema.getType())) {
                throw new AvroRuntimeException("Not a record schema: " + schema);
            }
            this.schema = schema;
            this.values = new Object[schema.getFields().size()];
        }

        public Record(Record other, boolean deepCopy) {
            this.schema = other.schema;
            this.values = new Object[this.schema.getFields().size()];
            if (deepCopy) {
                for (int ii = 0; ii < this.values.length; ii++) {
                    this.values[ii] = GenericData.INSTANCE.deepCopy(this.schema.getFields().get(ii).schema(), other.values[ii]);
                }
                return;
            }
            System.arraycopy(other.values, 0, this.values, 0, other.values.length);
        }

        @Override // com.flurry.org.apache.avro.generic.GenericContainer
        public Schema getSchema() {
            return this.schema;
        }

        @Override // com.flurry.org.apache.avro.generic.GenericRecord
        public void put(String key, Object value) {
            Schema.Field field = this.schema.getField(key);
            if (field == null) {
                throw new AvroRuntimeException("Not a valid schema field: " + key);
            }
            this.values[field.pos()] = value;
        }

        @Override // com.flurry.org.apache.avro.generic.IndexedRecord
        public void put(int i, Object v) {
            this.values[i] = v;
        }

        @Override // com.flurry.org.apache.avro.generic.GenericRecord
        public Object get(String key) {
            Schema.Field field = this.schema.getField(key);
            if (field == null) {
                return null;
            }
            return this.values[field.pos()];
        }

        @Override // com.flurry.org.apache.avro.generic.IndexedRecord
        public Object get(int i) {
            return this.values[i];
        }

        public boolean equals(Object o) {
            if (o == this) {
                return true;
            }
            if (o instanceof Record) {
                Record that = (Record) o;
                return this.schema.getFullName().equals(that.schema.getFullName()) && GenericData.get().compare(this, that, this.schema, true) == 0;
            }
            return false;
        }

        public int hashCode() {
            return GenericData.get().hashCode(this, this.schema);
        }

        @Override // java.lang.Comparable
        public int compareTo(Record that) {
            return GenericData.get().compare(this, that, this.schema);
        }

        public String toString() {
            return GenericData.get().toString(this);
        }
    }

    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public static class Array<T> extends AbstractList<T> implements GenericArray<T>, Comparable<GenericArray<T>> {
        private static final Object[] EMPTY = new Object[0];
        private Object[] elements;
        private final Schema schema;
        private int size;

        @Override // java.lang.Comparable
        public /* bridge */ /* synthetic */ int compareTo(Object x0) {
            return compareTo((GenericArray) ((GenericArray) x0));
        }

        public Array(int capacity, Schema schema) {
            this.elements = EMPTY;
            if (schema == null || !Schema.Type.ARRAY.equals(schema.getType())) {
                throw new AvroRuntimeException("Not an array schema: " + schema);
            }
            this.schema = schema;
            if (capacity != 0) {
                this.elements = new Object[capacity];
            }
        }

        public Array(Schema schema, Collection<T> c) {
            this.elements = EMPTY;
            if (schema == null || !Schema.Type.ARRAY.equals(schema.getType())) {
                throw new AvroRuntimeException("Not an array schema: " + schema);
            }
            this.schema = schema;
            if (c != null) {
                this.elements = new Object[c.size()];
                addAll(c);
            }
        }

        @Override // com.flurry.org.apache.avro.generic.GenericContainer
        public Schema getSchema() {
            return this.schema;
        }

        @Override // java.util.AbstractCollection, java.util.Collection, java.util.List
        public int size() {
            return this.size;
        }

        @Override // java.util.AbstractList, java.util.AbstractCollection, java.util.Collection, java.util.List
        public void clear() {
            this.size = 0;
        }

        @Override // java.util.AbstractList, java.util.AbstractCollection, java.util.Collection, java.lang.Iterable, java.util.List
        public Iterator<T> iterator() {
            return new Iterator<T>() { // from class: com.flurry.org.apache.avro.generic.GenericData.Array.1
                private int position = 0;

                @Override // java.util.Iterator
                public boolean hasNext() {
                    return this.position < Array.this.size;
                }

                @Override // java.util.Iterator
                public T next() {
                    Object[] objArr = Array.this.elements;
                    int i = this.position;
                    this.position = i + 1;
                    return (T) objArr[i];
                }

                @Override // java.util.Iterator
                public void remove() {
                    throw new UnsupportedOperationException();
                }
            };
        }

        @Override // java.util.AbstractList, java.util.List
        public T get(int i) {
            if (i >= this.size) {
                throw new IndexOutOfBoundsException("Index " + i + " out of bounds.");
            }
            return (T) this.elements[i];
        }

        @Override // java.util.AbstractList, java.util.AbstractCollection, java.util.Collection, java.util.List
        public boolean add(T o) {
            if (this.size == this.elements.length) {
                Object[] newElements = new Object[((this.size * 3) / 2) + 1];
                System.arraycopy(this.elements, 0, newElements, 0, this.size);
                this.elements = newElements;
            }
            Object[] objArr = this.elements;
            int i = this.size;
            this.size = i + 1;
            objArr[i] = o;
            return true;
        }

        @Override // java.util.AbstractList, java.util.List
        public void add(int location, T o) {
            if (location > this.size || location < 0) {
                throw new IndexOutOfBoundsException("Index " + location + " out of bounds.");
            }
            if (this.size == this.elements.length) {
                Object[] newElements = new Object[((this.size * 3) / 2) + 1];
                System.arraycopy(this.elements, 0, newElements, 0, this.size);
                this.elements = newElements;
            }
            System.arraycopy(this.elements, location, this.elements, location + 1, this.size - location);
            this.elements[location] = o;
            this.size++;
        }

        @Override // java.util.AbstractList, java.util.List
        public T set(int i, T o) {
            if (i >= this.size) {
                throw new IndexOutOfBoundsException("Index " + i + " out of bounds.");
            }
            T response = (T) this.elements[i];
            this.elements[i] = o;
            return response;
        }

        @Override // java.util.AbstractList, java.util.List
        public T remove(int i) {
            if (i >= this.size) {
                throw new IndexOutOfBoundsException("Index " + i + " out of bounds.");
            }
            T result = (T) this.elements[i];
            this.size--;
            System.arraycopy(this.elements, i + 1, this.elements, i, this.size - i);
            this.elements[this.size] = null;
            return result;
        }

        @Override // com.flurry.org.apache.avro.generic.GenericArray
        public T peek() {
            if (this.size < this.elements.length) {
                return (T) this.elements[this.size];
            }
            return null;
        }

        public int compareTo(GenericArray<T> that) {
            return GenericData.get().compare(this, that, getSchema());
        }

        @Override // com.flurry.org.apache.avro.generic.GenericArray
        public void reverse() {
            int left = 0;
            for (int right = this.elements.length - 1; left < right; right--) {
                Object tmp = this.elements[left];
                this.elements[left] = this.elements[right];
                this.elements[right] = tmp;
                left++;
            }
        }

        @Override // java.util.AbstractCollection
        public String toString() {
            StringBuffer buffer = new StringBuffer();
            buffer.append("[");
            int count = 0;
            Iterator i$ = iterator();
            while (i$.hasNext()) {
                T e = i$.next();
                buffer.append(e == null ? DataFileConstants.NULL_CODEC : e.toString());
                count++;
                if (count < size()) {
                    buffer.append(", ");
                }
            }
            buffer.append("]");
            return buffer.toString();
        }
    }

    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public static class Fixed implements GenericFixed, Comparable<Fixed> {
        private byte[] bytes;
        private Schema schema;

        public Fixed(Schema schema) {
            setSchema(schema);
        }

        public Fixed(Schema schema, byte[] bytes) {
            this.schema = schema;
            this.bytes = bytes;
        }

        /* JADX INFO: Access modifiers changed from: protected */
        public Fixed() {
        }

        /* JADX INFO: Access modifiers changed from: protected */
        public void setSchema(Schema schema) {
            this.schema = schema;
            this.bytes = new byte[schema.getFixedSize()];
        }

        @Override // com.flurry.org.apache.avro.generic.GenericContainer
        public Schema getSchema() {
            return this.schema;
        }

        public void bytes(byte[] bytes) {
            this.bytes = bytes;
        }

        @Override // com.flurry.org.apache.avro.generic.GenericFixed
        public byte[] bytes() {
            return this.bytes;
        }

        public boolean equals(Object o) {
            if (o == this) {
                return true;
            }
            return (o instanceof GenericFixed) && Arrays.equals(this.bytes, ((GenericFixed) o).bytes());
        }

        public int hashCode() {
            return Arrays.hashCode(this.bytes);
        }

        public String toString() {
            return Arrays.toString(this.bytes);
        }

        @Override // java.lang.Comparable
        public int compareTo(Fixed that) {
            return BinaryData.compareBytes(this.bytes, 0, this.bytes.length, that.bytes, 0, that.bytes.length);
        }
    }

    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public static class EnumSymbol implements GenericEnumSymbol {
        private Schema schema;
        private String symbol;

        public EnumSymbol(Schema schema, String symbol) {
            this.schema = schema;
            this.symbol = symbol;
        }

        @Override // com.flurry.org.apache.avro.generic.GenericContainer
        public Schema getSchema() {
            return this.schema;
        }

        public boolean equals(Object o) {
            if (o == this) {
                return true;
            }
            return (o instanceof GenericEnumSymbol) && this.symbol.equals(o.toString());
        }

        public int hashCode() {
            return this.symbol.hashCode();
        }

        @Override // com.flurry.org.apache.avro.generic.GenericEnumSymbol
        public String toString() {
            return this.symbol;
        }
    }

    public DatumReader createDatumReader(Schema schema) {
        return new GenericDatumReader(schema, schema, this);
    }

    /* JADX INFO: Access modifiers changed from: package-private */
    /* renamed from: com.flurry.org.apache.avro.generic.GenericData$1  reason: invalid class name */
    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public static /* synthetic */ class AnonymousClass1 {
        static final /* synthetic */ int[] $SwitchMap$org$apache$avro$Schema$Type = new int[Schema.Type.values().length];

        static {
            try {
                $SwitchMap$org$apache$avro$Schema$Type[Schema.Type.RECORD.ordinal()] = 1;
            } catch (NoSuchFieldError e) {
            }
            try {
                $SwitchMap$org$apache$avro$Schema$Type[Schema.Type.ENUM.ordinal()] = 2;
            } catch (NoSuchFieldError e2) {
            }
            try {
                $SwitchMap$org$apache$avro$Schema$Type[Schema.Type.ARRAY.ordinal()] = 3;
            } catch (NoSuchFieldError e3) {
            }
            try {
                $SwitchMap$org$apache$avro$Schema$Type[Schema.Type.MAP.ordinal()] = 4;
            } catch (NoSuchFieldError e4) {
            }
            try {
                $SwitchMap$org$apache$avro$Schema$Type[Schema.Type.UNION.ordinal()] = 5;
            } catch (NoSuchFieldError e5) {
            }
            try {
                $SwitchMap$org$apache$avro$Schema$Type[Schema.Type.FIXED.ordinal()] = 6;
            } catch (NoSuchFieldError e6) {
            }
            try {
                $SwitchMap$org$apache$avro$Schema$Type[Schema.Type.STRING.ordinal()] = 7;
            } catch (NoSuchFieldError e7) {
            }
            try {
                $SwitchMap$org$apache$avro$Schema$Type[Schema.Type.BYTES.ordinal()] = 8;
            } catch (NoSuchFieldError e8) {
            }
            try {
                $SwitchMap$org$apache$avro$Schema$Type[Schema.Type.INT.ordinal()] = 9;
            } catch (NoSuchFieldError e9) {
            }
            try {
                $SwitchMap$org$apache$avro$Schema$Type[Schema.Type.LONG.ordinal()] = 10;
            } catch (NoSuchFieldError e10) {
            }
            try {
                $SwitchMap$org$apache$avro$Schema$Type[Schema.Type.FLOAT.ordinal()] = 11;
            } catch (NoSuchFieldError e11) {
            }
            try {
                $SwitchMap$org$apache$avro$Schema$Type[Schema.Type.DOUBLE.ordinal()] = 12;
            } catch (NoSuchFieldError e12) {
            }
            try {
                $SwitchMap$org$apache$avro$Schema$Type[Schema.Type.BOOLEAN.ordinal()] = 13;
            } catch (NoSuchFieldError e13) {
            }
            try {
                $SwitchMap$org$apache$avro$Schema$Type[Schema.Type.NULL.ordinal()] = 14;
            } catch (NoSuchFieldError e14) {
            }
        }
    }

    public boolean validate(Schema schema, Object datum) {
        switch (AnonymousClass1.$SwitchMap$org$apache$avro$Schema$Type[schema.getType().ordinal()]) {
            case 1:
                if (isRecord(datum)) {
                    for (Schema.Field f : schema.getFields()) {
                        if (!validate(f.schema(), getField(datum, f.name(), f.pos()))) {
                            return false;
                        }
                    }
                    return true;
                }
                return false;
            case 2:
                return schema.getEnumSymbols().contains(datum.toString());
            case 3:
                if (datum instanceof Collection) {
                    for (Object element : (Collection) datum) {
                        if (!validate(schema.getElementType(), element)) {
                            return false;
                        }
                    }
                    return true;
                }
                return false;
            case 4:
                if (datum instanceof Map) {
                    Map<Object, Object> map = (Map) datum;
                    for (Map.Entry<Object, Object> entry : map.entrySet()) {
                        if (!validate(schema.getValueType(), entry.getValue())) {
                            return false;
                        }
                    }
                    return true;
                }
                return false;
            case 5:
                for (Schema type : schema.getTypes()) {
                    if (validate(type, datum)) {
                        return true;
                    }
                }
                return false;
            case 6:
                return ((datum instanceof GenericFixed) && ((GenericFixed) datum).bytes().length == schema.getFixedSize()) ? false : false;
            case IabHelper.BILLING_RESPONSE_RESULT_ITEM_ALREADY_OWNED /* 7 */:
                return isString(datum);
            case 8:
                return isBytes(datum);
            case 9:
                return datum instanceof Integer;
            case TapjoyVideoObject.BUTTON_MAX /* 10 */:
                return datum instanceof Long;
            case 11:
                return datum instanceof Float;
            case 12:
                return datum instanceof Double;
            case 13:
                return datum instanceof Boolean;
            case 14:
                return datum == null;
            default:
                return false;
        }
    }

    public String toString(Object datum) {
        StringBuilder buffer = new StringBuilder();
        toString(datum, buffer);
        return buffer.toString();
    }

    protected void toString(Object datum, StringBuilder buffer) {
        if (isRecord(datum)) {
            buffer.append("{");
            int count = 0;
            Schema schema = getRecordSchema(datum);
            for (Schema.Field f : schema.getFields()) {
                toString(f.name(), buffer);
                buffer.append(": ");
                toString(getField(datum, f.name(), f.pos()), buffer);
                count++;
                if (count < schema.getFields().size()) {
                    buffer.append(", ");
                }
            }
            buffer.append("}");
        } else if (datum instanceof Collection) {
            Collection<?> array = (Collection) datum;
            buffer.append("[");
            long last = array.size() - 1;
            int i = 0;
            for (Object element : array) {
                toString(element, buffer);
                int i2 = i + 1;
                if (i < last) {
                    buffer.append(", ");
                }
                i = i2;
            }
            buffer.append("]");
        } else if (datum instanceof Map) {
            buffer.append("{");
            int count2 = 0;
            Map<Object, Object> map = (Map) datum;
            for (Map.Entry<Object, Object> entry : map.entrySet()) {
                toString(entry.getKey(), buffer);
                buffer.append(": ");
                toString(entry.getValue(), buffer);
                count2++;
                if (count2 < map.size()) {
                    buffer.append(", ");
                }
            }
            buffer.append("}");
        } else if ((datum instanceof CharSequence) || (datum instanceof GenericEnumSymbol)) {
            buffer.append("\"");
            writeEscapedString(datum.toString(), buffer);
            buffer.append("\"");
        } else if (datum instanceof ByteBuffer) {
            buffer.append("{\"bytes\": \"");
            ByteBuffer bytes = (ByteBuffer) datum;
            for (int i3 = bytes.position(); i3 < bytes.limit(); i3++) {
                buffer.append((char) bytes.get(i3));
            }
            buffer.append("\"}");
        } else {
            buffer.append(datum);
        }
    }

    private void writeEscapedString(String string, StringBuilder builder) {
        for (int i = 0; i < string.length(); i++) {
            char ch = string.charAt(i);
            switch (ch) {
                case '\b':
                    builder.append("\\b");
                    break;
                case '\t':
                    builder.append("\\t");
                    break;
                case TapjoyVideoObject.BUTTON_MAX /* 10 */:
                    builder.append("\\n");
                    break;
                case '\f':
                    builder.append("\\f");
                    break;
                case '\r':
                    builder.append("\\r");
                    break;
                case '\"':
                    builder.append("\\\"");
                    break;
                case '/':
                    builder.append("\\/");
                    break;
                case '\\':
                    builder.append("\\\\");
                    break;
                default:
                    if ((ch >= 0 && ch <= 31) || ((ch >= 127 && ch <= 159) || (ch >= 8192 && ch <= 8447))) {
                        Integer.toHexString(ch);
                        builder.append("\\u");
                        for (int j = 0; j < 4 - builder.length(); j++) {
                            builder.append('0');
                        }
                        builder.append(string.toUpperCase());
                        break;
                    } else {
                        builder.append(ch);
                        break;
                    }
            }
        }
    }

    public Schema induce(Object datum) {
        if (isRecord(datum)) {
            return getRecordSchema(datum);
        }
        if (datum instanceof Collection) {
            Schema elementType = null;
            for (Object element : (Collection) datum) {
                if (elementType == null) {
                    elementType = induce(element);
                } else if (!elementType.equals(induce(element))) {
                    throw new AvroTypeException("No mixed type arrays.");
                }
            }
            if (elementType == null) {
                throw new AvroTypeException("Empty array: " + datum);
            }
            return Schema.createArray(elementType);
        } else if (datum instanceof Map) {
            Map<Object, Object> map = (Map) datum;
            Schema value = null;
            for (Map.Entry<Object, Object> entry : map.entrySet()) {
                if (value == null) {
                    value = induce(entry.getValue());
                } else if (!value.equals(induce(entry.getValue()))) {
                    throw new AvroTypeException("No mixed type map values.");
                }
            }
            if (value == null) {
                throw new AvroTypeException("Empty map: " + datum);
            }
            return Schema.createMap(value);
        } else if (datum instanceof GenericFixed) {
            return Schema.createFixed(null, null, null, ((GenericFixed) datum).bytes().length);
        } else {
            if (datum instanceof CharSequence) {
                return Schema.create(Schema.Type.STRING);
            }
            if (datum instanceof ByteBuffer) {
                return Schema.create(Schema.Type.BYTES);
            }
            if (datum instanceof Integer) {
                return Schema.create(Schema.Type.INT);
            }
            if (datum instanceof Long) {
                return Schema.create(Schema.Type.LONG);
            }
            if (datum instanceof Float) {
                return Schema.create(Schema.Type.FLOAT);
            }
            if (datum instanceof Double) {
                return Schema.create(Schema.Type.DOUBLE);
            }
            if (datum instanceof Boolean) {
                return Schema.create(Schema.Type.BOOLEAN);
            }
            if (datum == null) {
                return Schema.create(Schema.Type.NULL);
            }
            throw new AvroTypeException("Can't create schema for: " + datum);
        }
    }

    public void setField(Object record, String name, int position, Object o) {
        ((IndexedRecord) record).put(position, o);
    }

    public Object getField(Object record, String name, int position) {
        return ((IndexedRecord) record).get(position);
    }

    /* JADX INFO: Access modifiers changed from: protected */
    public Object getRecordState(Object record, Schema schema) {
        return null;
    }

    /* JADX INFO: Access modifiers changed from: protected */
    public void setField(Object r, String n, int p, Object o, Object state) {
        setField(r, n, p, o);
    }

    /* JADX INFO: Access modifiers changed from: protected */
    public Object getField(Object record, String name, int pos, Object state) {
        return getField(record, name, pos);
    }

    public int resolveUnion(Schema union, Object datum) {
        Integer i = union.getIndexNamed(getSchemaName(datum));
        if (i != null) {
            return i.intValue();
        }
        throw new UnresolvedUnionException(union, datum);
    }

    protected String getSchemaName(Object datum) {
        if (datum == null) {
            return Schema.Type.NULL.getName();
        }
        if (isRecord(datum)) {
            return getRecordSchema(datum).getFullName();
        }
        if (isEnum(datum)) {
            return getEnumSchema(datum).getFullName();
        }
        if (isArray(datum)) {
            return Schema.Type.ARRAY.getName();
        }
        if (isMap(datum)) {
            return Schema.Type.MAP.getName();
        }
        if (isFixed(datum)) {
            return getFixedSchema(datum).getFullName();
        }
        if (isString(datum)) {
            return Schema.Type.STRING.getName();
        }
        if (isBytes(datum)) {
            return Schema.Type.BYTES.getName();
        }
        if (datum instanceof Integer) {
            return Schema.Type.INT.getName();
        }
        if (datum instanceof Long) {
            return Schema.Type.LONG.getName();
        }
        if (datum instanceof Float) {
            return Schema.Type.FLOAT.getName();
        }
        if (datum instanceof Double) {
            return Schema.Type.DOUBLE.getName();
        }
        if (datum instanceof Boolean) {
            return Schema.Type.BOOLEAN.getName();
        }
        throw new AvroRuntimeException("Unknown datum type: " + datum);
    }

    protected boolean instanceOf(Schema schema, Object datum) {
        switch (AnonymousClass1.$SwitchMap$org$apache$avro$Schema$Type[schema.getType().ordinal()]) {
            case 1:
                if (isRecord(datum)) {
                    if (schema.getFullName() != null) {
                        r0 = schema.getFullName().equals(getRecordSchema(datum).getFullName());
                    } else if (getRecordSchema(datum).getFullName() != null) {
                        r0 = false;
                    }
                    return r0;
                }
                return false;
            case 2:
                if (isEnum(datum)) {
                    return schema.getFullName().equals(getEnumSchema(datum).getFullName());
                }
                return false;
            case 3:
                return isArray(datum);
            case 4:
                return isMap(datum);
            case 5:
            default:
                throw new AvroRuntimeException("Unexpected type: " + schema);
            case 6:
                if (isFixed(datum)) {
                    return schema.getFullName().equals(getFixedSchema(datum).getFullName());
                }
                return false;
            case IabHelper.BILLING_RESPONSE_RESULT_ITEM_ALREADY_OWNED /* 7 */:
                return isString(datum);
            case 8:
                return isBytes(datum);
            case 9:
                return datum instanceof Integer;
            case TapjoyVideoObject.BUTTON_MAX /* 10 */:
                return datum instanceof Long;
            case 11:
                return datum instanceof Float;
            case 12:
                return datum instanceof Double;
            case 13:
                return datum instanceof Boolean;
            case 14:
                return datum == null;
        }
    }

    protected boolean isArray(Object datum) {
        return datum instanceof Collection;
    }

    /* JADX INFO: Access modifiers changed from: protected */
    public boolean isRecord(Object datum) {
        return datum instanceof IndexedRecord;
    }

    /* JADX INFO: Access modifiers changed from: protected */
    public Schema getRecordSchema(Object record) {
        return ((GenericContainer) record).getSchema();
    }

    /* JADX INFO: Access modifiers changed from: protected */
    public boolean isEnum(Object datum) {
        return datum instanceof GenericEnumSymbol;
    }

    /* JADX INFO: Access modifiers changed from: protected */
    public Schema getEnumSchema(Object enu) {
        return ((GenericContainer) enu).getSchema();
    }

    protected boolean isMap(Object datum) {
        return datum instanceof Map;
    }

    protected boolean isFixed(Object datum) {
        return datum instanceof GenericFixed;
    }

    protected Schema getFixedSchema(Object fixed) {
        return ((GenericContainer) fixed).getSchema();
    }

    protected boolean isString(Object datum) {
        return datum instanceof CharSequence;
    }

    /* JADX INFO: Access modifiers changed from: protected */
    public boolean isBytes(Object datum) {
        return datum instanceof ByteBuffer;
    }

    public int hashCode(Object o, Schema s) {
        if (o == null) {
            return 0;
        }
        int hashCode = 1;
        switch (AnonymousClass1.$SwitchMap$org$apache$avro$Schema$Type[s.getType().ordinal()]) {
            case 1:
                for (Schema.Field f : s.getFields()) {
                    if (f.order() != Schema.Field.Order.IGNORE) {
                        hashCode = hashCodeAdd(hashCode, getField(o, f.name(), f.pos()), f.schema());
                    }
                }
                return hashCode;
            case 2:
                return s.getEnumOrdinal(o.toString());
            case 3:
                Collection<?> a = (Collection) o;
                Schema elementType = s.getElementType();
                for (Object e : a) {
                    hashCode = hashCodeAdd(hashCode, e, elementType);
                }
                return hashCode;
            case 4:
            case 6:
            case 8:
            case 9:
            case TapjoyVideoObject.BUTTON_MAX /* 10 */:
            case 11:
            case 12:
            case 13:
            default:
                return o.hashCode();
            case 5:
                return hashCode(o, s.getTypes().get(resolveUnion(s, o)));
            case IabHelper.BILLING_RESPONSE_RESULT_ITEM_ALREADY_OWNED /* 7 */:
                if (!(o instanceof Utf8)) {
                    o = new Utf8(o.toString());
                }
                return o.hashCode();
            case 14:
                return 0;
        }
    }

    protected int hashCodeAdd(int hashCode, Object o, Schema s) {
        return (hashCode * 31) + hashCode(o, s);
    }

    public int compare(Object o1, Object o2, Schema s) {
        return compare(o1, o2, s, false);
    }

    /* JADX INFO: Access modifiers changed from: protected */
    public int compare(Object o1, Object o2, Schema s, boolean equals) {
        if (o1 == o2) {
            return 0;
        }
        switch (AnonymousClass1.$SwitchMap$org$apache$avro$Schema$Type[s.getType().ordinal()]) {
            case 1:
                for (Schema.Field f : s.getFields()) {
                    if (f.order() != Schema.Field.Order.IGNORE) {
                        int pos = f.pos();
                        String name = f.name();
                        int compare = compare(getField(o1, name, pos), getField(o2, name, pos), f.schema(), equals);
                        if (compare != 0) {
                            if (f.order() == Schema.Field.Order.DESCENDING) {
                                compare = -compare;
                            }
                            return compare;
                        }
                    }
                }
                return 0;
            case 2:
                return s.getEnumOrdinal(o1.toString()) - s.getEnumOrdinal(o2.toString());
            case 3:
                Collection a1 = (Collection) o1;
                Collection a2 = (Collection) o2;
                Iterator e1 = a1.iterator();
                Iterator e2 = a2.iterator();
                Schema elementType = s.getElementType();
                while (e1.hasNext() && e2.hasNext()) {
                    int compare2 = compare(e1.next(), e2.next(), elementType, equals);
                    if (compare2 != 0) {
                        return compare2;
                    }
                }
                if (e1.hasNext()) {
                    return 1;
                }
                return e2.hasNext() ? -1 : 0;
            case 4:
                if (equals) {
                    return ((Map) o1).equals(o2) ? 0 : 1;
                }
                throw new AvroRuntimeException("Can't compare maps!");
            case 5:
                int i1 = resolveUnion(s, o1);
                int i2 = resolveUnion(s, o2);
                return i1 == i2 ? compare(o1, o2, s.getTypes().get(i1), equals) : i1 - i2;
            case 6:
            case 8:
            case 9:
            case TapjoyVideoObject.BUTTON_MAX /* 10 */:
            case 11:
            case 12:
            case 13:
            default:
                return ((Comparable) o1).compareTo(o2);
            case IabHelper.BILLING_RESPONSE_RESULT_ITEM_ALREADY_OWNED /* 7 */:
                Utf8 u1 = o1 instanceof Utf8 ? (Utf8) o1 : new Utf8(o1.toString());
                Utf8 u2 = o2 instanceof Utf8 ? (Utf8) o2 : new Utf8(o2.toString());
                return u1.compareTo(u2);
            case 14:
                return 0;
        }
    }

    public Object deepCopy(Schema schema, Object value) {
        if (value == null) {
            return null;
        }
        switch (AnonymousClass1.$SwitchMap$org$apache$avro$Schema$Type[schema.getType().ordinal()]) {
            case 1:
                IndexedRecord recordValue = (IndexedRecord) value;
                IndexedRecord recordCopy = (IndexedRecord) newRecord(null, schema);
                for (Schema.Field field : schema.getFields()) {
                    recordCopy.put(field.pos(), deepCopy(field.schema(), recordValue.get(field.pos())));
                }
                return recordCopy;
            case 2:
                return value;
            case 3:
                List<Object> arrayValue = (List) value;
                List<Object> arrayCopy = new Array<>(arrayValue.size(), schema);
                for (Object obj : arrayValue) {
                    arrayCopy.add(deepCopy(schema.getElementType(), obj));
                }
                return arrayCopy;
            case 4:
                Map<CharSequence, Object> mapValue = (Map) value;
                Map<CharSequence, Object> mapCopy = new HashMap<>(mapValue.size());
                for (Map.Entry<CharSequence, Object> entry : mapValue.entrySet()) {
                    mapCopy.put((CharSequence) deepCopy(STRINGS, entry.getKey()), deepCopy(schema.getValueType(), entry.getValue()));
                }
                return mapCopy;
            case 5:
                return deepCopy(schema.getTypes().get(resolveUnion(schema, value)), value);
            case 6:
                return createFixed(null, ((GenericFixed) value).bytes(), schema);
            case IabHelper.BILLING_RESPONSE_RESULT_ITEM_ALREADY_OWNED /* 7 */:
                if (!(value instanceof String)) {
                    if (value instanceof Utf8) {
                        return new Utf8((Utf8) value);
                    }
                    return new Utf8(value.toString());
                }
                return value;
            case 8:
                ByteBuffer byteBufferValue = (ByteBuffer) value;
                byte[] bytesCopy = new byte[byteBufferValue.capacity()];
                byteBufferValue.rewind();
                byteBufferValue.get(bytesCopy);
                byteBufferValue.rewind();
                return ByteBuffer.wrap(bytesCopy);
            case 9:
                return new Integer(((Integer) value).intValue());
            case TapjoyVideoObject.BUTTON_MAX /* 10 */:
                return new Long(((Long) value).longValue());
            case 11:
                return new Float(((Float) value).floatValue());
            case 12:
                return new Double(((Double) value).doubleValue());
            case 13:
                return new Boolean(((Boolean) value).booleanValue());
            case 14:
                return null;
            default:
                throw new AvroRuntimeException("Deep copy failed for schema \"" + schema + "\" and value \"" + value + "\"");
        }
    }

    public Object createFixed(Object old, Schema schema) {
        return ((old instanceof GenericFixed) && ((GenericFixed) old).bytes().length == schema.getFixedSize()) ? old : new Fixed(schema);
    }

    public Object createFixed(Object old, byte[] bytes, Schema schema) {
        GenericFixed fixed = (GenericFixed) createFixed(old, schema);
        System.arraycopy(bytes, 0, fixed.bytes(), 0, schema.getFixedSize());
        return fixed;
    }

    public Object newRecord(Object old, Schema schema) {
        if (old instanceof IndexedRecord) {
            IndexedRecord record = (IndexedRecord) old;
            if (record.getSchema() == schema) {
                return record;
            }
        }
        return new Record(schema);
    }
}
