package com.flurry.org.apache.avro.file;

import com.flurry.org.apache.avro.AvroRuntimeException;
import com.flurry.org.apache.avro.Schema;
import com.flurry.org.apache.avro.io.BinaryDecoder;
import com.flurry.org.apache.avro.io.BinaryEncoder;
import com.flurry.org.apache.avro.io.DatumReader;
import com.flurry.org.apache.avro.io.DecoderFactory;
import java.io.Closeable;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;

/* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
public class DataFileStream<D> implements Iterator<D>, Iterable<D>, Closeable {
    private boolean availableBlock;
    private DataBlock block;
    ByteBuffer blockBuffer;
    long blockCount;
    long blockRemaining;
    private long blockSize;
    private Codec codec;
    BinaryDecoder datumIn;
    private Header header;
    private DatumReader<D> reader;
    byte[] syncBuffer;
    BinaryDecoder vin;

    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public static final class Header {
        Map<String, byte[]> meta;
        private transient List<String> metaKeyList;
        Schema schema;
        byte[] sync;

        private Header() {
            this.meta = new HashMap();
            this.metaKeyList = new ArrayList();
            this.sync = new byte[16];
        }
    }

    public DataFileStream(InputStream in, DatumReader<D> reader) throws IOException {
        this.availableBlock = false;
        this.datumIn = null;
        this.syncBuffer = new byte[16];
        this.block = null;
        this.reader = reader;
        initialize(in);
    }

    /* JADX INFO: Access modifiers changed from: protected */
    public DataFileStream(DatumReader<D> reader) throws IOException {
        this.availableBlock = false;
        this.datumIn = null;
        this.syncBuffer = new byte[16];
        this.block = null;
        this.reader = reader;
    }

    /* JADX INFO: Access modifiers changed from: package-private */
    public void initialize(InputStream in) throws IOException {
        this.header = new Header();
        this.vin = DecoderFactory.get().binaryDecoder(in, this.vin);
        byte[] magic = new byte[DataFileConstants.MAGIC.length];
        try {
            this.vin.readFixed(magic);
            if (!Arrays.equals(DataFileConstants.MAGIC, magic)) {
                throw new IOException("Not a data file.");
            }
            long l = this.vin.readMapStart();
            if (l > 0) {
                do {
                    for (long i = 0; i < l; i++) {
                        String key = this.vin.readString(null).toString();
                        ByteBuffer value = this.vin.readBytes(null);
                        byte[] bb = new byte[value.remaining()];
                        value.get(bb);
                        this.header.meta.put(key, bb);
                        this.header.metaKeyList.add(key);
                    }
                    l = this.vin.mapNext();
                } while (l != 0);
                this.vin.readFixed(this.header.sync);
                this.header.metaKeyList = Collections.unmodifiableList(this.header.metaKeyList);
                this.header.schema = Schema.parse(getMetaString(DataFileConstants.SCHEMA), false);
                this.codec = resolveCodec();
                this.reader.setSchema(this.header.schema);
            }
            this.vin.readFixed(this.header.sync);
            this.header.metaKeyList = Collections.unmodifiableList(this.header.metaKeyList);
            this.header.schema = Schema.parse(getMetaString(DataFileConstants.SCHEMA), false);
            this.codec = resolveCodec();
            this.reader.setSchema(this.header.schema);
        } catch (IOException e) {
            throw new IOException("Not a data file.");
        }
    }

    /* JADX INFO: Access modifiers changed from: package-private */
    public void initialize(InputStream in, Header header) throws IOException {
        this.header = header;
        this.codec = resolveCodec();
        this.reader.setSchema(header.schema);
    }

    /* JADX INFO: Access modifiers changed from: package-private */
    public Codec resolveCodec() {
        String codecStr = getMetaString(DataFileConstants.CODEC);
        return codecStr != null ? CodecFactory.fromString(codecStr).createInstance() : CodecFactory.nullCodec().createInstance();
    }

    public Header getHeader() {
        return this.header;
    }

    public Schema getSchema() {
        return this.header.schema;
    }

    public List<String> getMetaKeys() {
        return this.header.metaKeyList;
    }

    public byte[] getMeta(String key) {
        return this.header.meta.get(key);
    }

    public String getMetaString(String key) {
        byte[] value = getMeta(key);
        if (value == null) {
            return null;
        }
        try {
            return new String(value, "UTF-8");
        } catch (UnsupportedEncodingException e) {
            throw new RuntimeException(e);
        }
    }

    public long getMetaLong(String key) {
        return Long.parseLong(getMetaString(key));
    }

    @Override // java.lang.Iterable
    public Iterator<D> iterator() {
        return this;
    }

    @Override // java.util.Iterator
    public boolean hasNext() {
        try {
            if (this.blockRemaining == 0) {
                if (this.datumIn != null) {
                    boolean atEnd = this.datumIn.isEnd();
                    if (!atEnd) {
                        throw new IOException("Block read partially, the data may be corrupt");
                    }
                }
                if (hasNextBlock()) {
                    this.block = nextRawBlock(this.block);
                    this.block.decompressUsing(this.codec);
                    this.blockBuffer = this.block.getAsByteBuffer();
                    this.datumIn = DecoderFactory.get().binaryDecoder(this.blockBuffer.array(), this.blockBuffer.arrayOffset() + this.blockBuffer.position(), this.blockBuffer.remaining(), this.datumIn);
                }
            }
            return this.blockRemaining != 0;
        } catch (EOFException e) {
            return false;
        } catch (IOException e2) {
            throw new AvroRuntimeException(e2);
        }
    }

    @Override // java.util.Iterator
    public D next() {
        try {
            return next(null);
        } catch (IOException e) {
            throw new AvroRuntimeException(e);
        }
    }

    public D next(D reuse) throws IOException {
        if (!hasNext()) {
            throw new NoSuchElementException();
        }
        D result = this.reader.read(reuse, this.datumIn);
        long j = this.blockRemaining - 1;
        this.blockRemaining = j;
        if (0 == j) {
            blockFinished();
        }
        return result;
    }

    public ByteBuffer nextBlock() throws IOException {
        if (!hasNext()) {
            throw new NoSuchElementException();
        }
        if (this.blockRemaining != this.blockCount) {
            throw new IllegalStateException("Not at block start.");
        }
        this.blockRemaining = 0L;
        this.datumIn = null;
        return this.blockBuffer;
    }

    public long getBlockCount() {
        return this.blockCount;
    }

    protected void blockFinished() throws IOException {
    }

    /* JADX INFO: Access modifiers changed from: package-private */
    public boolean hasNextBlock() {
        try {
            if (this.availableBlock) {
                return true;
            }
            if (this.vin.isEnd()) {
                return false;
            }
            this.blockRemaining = this.vin.readLong();
            this.blockSize = this.vin.readLong();
            if (this.blockSize > 2147483647L || this.blockSize < 0) {
                throw new IOException("Block size invalid or too large for this implementation: " + this.blockSize);
            }
            this.blockCount = this.blockRemaining;
            this.availableBlock = true;
            return true;
        } catch (EOFException e) {
            return false;
        } catch (IOException e2) {
            throw new AvroRuntimeException(e2);
        }
    }

    /* JADX INFO: Access modifiers changed from: package-private */
    public DataBlock nextRawBlock(DataBlock reuse) throws IOException {
        if (!hasNextBlock()) {
            throw new NoSuchElementException();
        }
        if (reuse == null || reuse.data.length < ((int) this.blockSize)) {
            reuse = new DataBlock(this.blockRemaining, (int) this.blockSize);
        } else {
            reuse.numEntries = this.blockRemaining;
            reuse.blockSize = (int) this.blockSize;
        }
        this.vin.readFixed(reuse.data, 0, reuse.blockSize);
        this.vin.readFixed(this.syncBuffer);
        if (!Arrays.equals(this.syncBuffer, this.header.sync)) {
            throw new IOException("Invalid sync!");
        }
        this.availableBlock = false;
        return reuse;
    }

    @Override // java.util.Iterator
    public void remove() {
        throw new UnsupportedOperationException();
    }

    @Override // java.io.Closeable, java.lang.AutoCloseable
    public void close() throws IOException {
        this.vin.inputStream().close();
    }

    /* JADX INFO: Access modifiers changed from: package-private */
    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public static class DataBlock {
        private int blockSize;
        private byte[] data;
        private long numEntries;
        private int offset;

        private DataBlock(long numEntries, int blockSize) {
            this.offset = 0;
            this.data = new byte[blockSize];
            this.numEntries = numEntries;
            this.blockSize = blockSize;
        }

        /* JADX INFO: Access modifiers changed from: package-private */
        public DataBlock(ByteBuffer block, long numEntries) {
            this.offset = 0;
            this.data = block.array();
            this.blockSize = block.remaining();
            this.offset = block.arrayOffset() + block.position();
            this.numEntries = numEntries;
        }

        byte[] getData() {
            return this.data;
        }

        long getNumEntries() {
            return this.numEntries;
        }

        int getBlockSize() {
            return this.blockSize;
        }

        ByteBuffer getAsByteBuffer() {
            return ByteBuffer.wrap(this.data, this.offset, this.blockSize);
        }

        /* JADX INFO: Access modifiers changed from: package-private */
        public void decompressUsing(Codec c) throws IOException {
            ByteBuffer result = c.decompress(getAsByteBuffer());
            this.data = result.array();
            this.blockSize = result.remaining();
        }

        /* JADX INFO: Access modifiers changed from: package-private */
        public void compressUsing(Codec c) throws IOException {
            ByteBuffer result = c.compress(getAsByteBuffer());
            this.data = result.array();
            this.blockSize = result.remaining();
        }

        /* JADX INFO: Access modifiers changed from: package-private */
        public void writeBlockTo(BinaryEncoder e, byte[] sync) throws IOException {
            e.writeLong(this.numEntries);
            e.writeLong(this.blockSize);
            e.writeFixed(this.data, this.offset, this.blockSize);
            e.writeFixed(sync);
            e.flush();
        }
    }
}
