package com.flurry.org.apache.avro.file;

import com.flurry.android.Constants;
import com.flurry.org.apache.avro.file.DataFileStream;
import com.flurry.org.apache.avro.io.DatumReader;
import com.flurry.org.apache.avro.io.DecoderFactory;
import java.io.EOFException;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.Arrays;

/* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
public class DataFileReader<D> extends DataFileStream<D> implements FileReader<D> {
    private long blockStart;
    private SeekableInputStream sin;

    public static <D> FileReader<D> openReader(File file, DatumReader<D> reader) throws IOException {
        return openReader(new SeekableFileInput(file), reader);
    }

    public static <D> FileReader<D> openReader(SeekableInput in, DatumReader<D> reader) throws IOException {
        if (in.length() < DataFileConstants.MAGIC.length) {
            throw new IOException("Not an Avro data file");
        }
        byte[] magic = new byte[DataFileConstants.MAGIC.length];
        in.seek(0L);
        int c = 0;
        while (c < magic.length) {
            c = in.read(magic, c, magic.length - c);
        }
        in.seek(0L);
        if (Arrays.equals(DataFileConstants.MAGIC, magic)) {
            return new DataFileReader(in, reader);
        }
        if (Arrays.equals(DataFileReader12.MAGIC, magic)) {
            return new DataFileReader12(in, reader);
        }
        throw new IOException("Not an Avro data file");
    }

    public static <D> DataFileReader<D> openReader(SeekableInput in, DatumReader<D> reader, DataFileStream.Header header, boolean sync) throws IOException {
        DataFileReader<D> dreader = new DataFileReader<>(in, reader, header);
        if (sync) {
            dreader.sync(in.tell());
        } else {
            dreader.seek(in.tell());
        }
        return dreader;
    }

    public DataFileReader(File file, DatumReader<D> reader) throws IOException {
        this(new SeekableFileInput(file), reader);
    }

    public DataFileReader(SeekableInput sin, DatumReader<D> reader) throws IOException {
        super(reader);
        this.sin = new SeekableInputStream(sin);
        initialize(this.sin);
        blockFinished();
    }

    protected DataFileReader(SeekableInput sin, DatumReader<D> reader, DataFileStream.Header header) throws IOException {
        super(reader);
        this.sin = new SeekableInputStream(sin);
        initialize(this.sin, header);
    }

    public void seek(long position) throws IOException {
        this.sin.seek(position);
        this.vin = DecoderFactory.get().binaryDecoder(this.sin, this.vin);
        this.datumIn = null;
        this.blockRemaining = 0L;
        this.blockStart = position;
    }

    @Override // com.flurry.org.apache.avro.file.FileReader
    public void sync(long position) throws IOException {
        seek(position);
        if (position == 0 && getMeta("avro.sync") != null) {
            initialize(this.sin);
            return;
        }
        int i = 0;
        try {
            InputStream in = this.vin.inputStream();
            this.vin.readFixed(this.syncBuffer);
            while (true) {
                int j = 0;
                while (j < 16 && getHeader().sync[j] == this.syncBuffer[(i + j) % 16]) {
                    j++;
                }
                if (j == 16) {
                    this.blockStart = i + position + 16;
                    return;
                }
                int b = in.read();
                int i2 = i + 1;
                try {
                    this.syncBuffer[i % 16] = (byte) b;
                    if (b == -1) {
                        break;
                    }
                    i = i2;
                } catch (EOFException e) {
                }
            }
        } catch (EOFException e2) {
        }
        this.blockStart = this.sin.tell();
    }

    @Override // com.flurry.org.apache.avro.file.DataFileStream
    protected void blockFinished() throws IOException {
        this.blockStart = this.sin.tell() - this.vin.inputStream().available();
    }

    public long previousSync() {
        return this.blockStart;
    }

    @Override // com.flurry.org.apache.avro.file.FileReader
    public boolean pastSync(long position) throws IOException {
        return this.blockStart >= 16 + position || this.blockStart >= this.sin.length();
    }

    @Override // com.flurry.org.apache.avro.file.FileReader
    public long tell() throws IOException {
        return this.sin.tell();
    }

    /* JADX INFO: Access modifiers changed from: package-private */
    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public static class SeekableInputStream extends InputStream implements SeekableInput {
        private SeekableInput in;
        private final byte[] oneByte = new byte[1];

        /* JADX INFO: Access modifiers changed from: package-private */
        public SeekableInputStream(SeekableInput in) throws IOException {
            this.in = in;
        }

        @Override // com.flurry.org.apache.avro.file.SeekableInput
        public void seek(long p) throws IOException {
            if (p < 0) {
                throw new IOException("Illegal seek: " + p);
            }
            this.in.seek(p);
        }

        @Override // com.flurry.org.apache.avro.file.SeekableInput
        public long tell() throws IOException {
            return this.in.tell();
        }

        @Override // com.flurry.org.apache.avro.file.SeekableInput
        public long length() throws IOException {
            return this.in.length();
        }

        @Override // java.io.InputStream
        public int read(byte[] b) throws IOException {
            return this.in.read(b, 0, b.length);
        }

        @Override // java.io.InputStream, com.flurry.org.apache.avro.file.SeekableInput
        public int read(byte[] b, int off, int len) throws IOException {
            return this.in.read(b, off, len);
        }

        @Override // java.io.InputStream
        public int read() throws IOException {
            int n = read(this.oneByte, 0, 1);
            if (n == 1) {
                return this.oneByte[0] & Constants.UNKNOWN;
            }
            return n;
        }

        @Override // java.io.InputStream
        public long skip(long skip) throws IOException {
            long position = this.in.tell();
            long length = this.in.length();
            long remaining = length - position;
            if (remaining > skip) {
                this.in.seek(skip);
                return this.in.tell() - position;
            }
            this.in.seek(remaining);
            return this.in.tell() - position;
        }

        @Override // java.io.InputStream, java.io.Closeable, java.lang.AutoCloseable
        public void close() throws IOException {
            this.in.close();
            super.close();
        }

        @Override // java.io.InputStream
        public int available() throws IOException {
            long remaining = this.in.length() - this.in.tell();
            if (remaining > 2147483647L) {
                return Integer.MAX_VALUE;
            }
            return (int) remaining;
        }
    }
}
