package com.flurry.org.apache.avro.data;

import com.flurry.org.apache.avro.AvroRuntimeException;
import com.flurry.org.apache.avro.Schema;
import com.flurry.org.apache.avro.generic.GenericData;
import com.flurry.org.apache.avro.generic.IndexedRecord;
import com.flurry.org.apache.avro.io.BinaryDecoder;
import com.flurry.org.apache.avro.io.BinaryEncoder;
import com.flurry.org.apache.avro.io.DecoderFactory;
import com.flurry.org.apache.avro.io.EncoderFactory;
import com.flurry.org.apache.avro.io.parsing.ResolvingGrammarGenerator;
import com.flurry.org.codehaus.jackson.JsonNode;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.Arrays;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

/* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
public abstract class RecordBuilderBase<T extends IndexedRecord> implements RecordBuilder<T> {
    private static final ConcurrentMap<String, ConcurrentMap<Integer, Object>> DEFAULT_VALUE_CACHE = new ConcurrentHashMap();
    private static final Schema.Field[] EMPTY_FIELDS = new Schema.Field[0];
    private final GenericData data;
    private final boolean[] fieldSetFlags;
    private final Schema.Field[] fields;
    private final Schema schema;
    private BinaryEncoder encoder = null;
    private BinaryDecoder decoder = null;

    /* JADX INFO: Access modifiers changed from: protected */
    public final Schema schema() {
        return this.schema;
    }

    /* JADX INFO: Access modifiers changed from: protected */
    public final Schema.Field[] fields() {
        return this.fields;
    }

    /* JADX INFO: Access modifiers changed from: protected */
    public final boolean[] fieldSetFlags() {
        return this.fieldSetFlags;
    }

    /* JADX INFO: Access modifiers changed from: protected */
    public final GenericData data() {
        return this.data;
    }

    /* JADX INFO: Access modifiers changed from: protected */
    public RecordBuilderBase(Schema schema, GenericData data) {
        this.schema = schema;
        this.data = data;
        this.fields = (Schema.Field[]) schema.getFields().toArray(EMPTY_FIELDS);
        this.fieldSetFlags = new boolean[this.fields.length];
    }

    /* JADX INFO: Access modifiers changed from: protected */
    public RecordBuilderBase(RecordBuilderBase<T> other, GenericData data) {
        this.schema = other.schema;
        this.data = data;
        this.fields = (Schema.Field[]) this.schema.getFields().toArray(EMPTY_FIELDS);
        this.fieldSetFlags = new boolean[other.fieldSetFlags.length];
        System.arraycopy(other.fieldSetFlags, 0, this.fieldSetFlags, 0, this.fieldSetFlags.length);
    }

    /* JADX INFO: Access modifiers changed from: protected */
    public void validate(Schema.Field field, Object value) {
        if (isValidValue(field, value) || field.defaultValue() != null) {
            return;
        }
        throw new AvroRuntimeException("Field " + field + " does not accept null values");
    }

    /* JADX INFO: Access modifiers changed from: protected */
    public static boolean isValidValue(Schema.Field f, Object value) {
        Schema schema;
        Schema.Type type;
        if (value == null && (type = (schema = f.schema()).getType()) != Schema.Type.NULL) {
            if (type == Schema.Type.UNION) {
                for (Schema s : schema.getTypes()) {
                    if (s.getType() == Schema.Type.NULL) {
                        return true;
                    }
                }
            }
            return false;
        }
        return true;
    }

    /* JADX INFO: Access modifiers changed from: protected */
    public Object defaultValue(Schema.Field field) throws IOException {
        JsonNode defaultJsonValue = field.defaultValue();
        if (defaultJsonValue == null) {
            throw new AvroRuntimeException("Field " + field + " not set and has no default value");
        }
        if (defaultJsonValue.isNull() && (field.schema().getType() == Schema.Type.NULL || (field.schema().getType() == Schema.Type.UNION && field.schema().getTypes().get(0).getType() == Schema.Type.NULL))) {
            return null;
        }
        ConcurrentMap<Integer, Object> defaultSchemaValues = DEFAULT_VALUE_CACHE.get(this.schema.getFullName());
        if (defaultSchemaValues == null) {
            DEFAULT_VALUE_CACHE.putIfAbsent(this.schema.getFullName(), new ConcurrentHashMap(this.fields.length));
            defaultSchemaValues = DEFAULT_VALUE_CACHE.get(this.schema.getFullName());
        }
        Object defaultValue = defaultSchemaValues.get(Integer.valueOf(field.pos()));
        if (defaultValue == null) {
            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            this.encoder = EncoderFactory.get().binaryEncoder(baos, this.encoder);
            ResolvingGrammarGenerator.encode(this.encoder, field.schema(), defaultJsonValue);
            this.encoder.flush();
            this.decoder = DecoderFactory.get().binaryDecoder(baos.toByteArray(), this.decoder);
            defaultValue = this.data.createDatumReader(field.schema()).read(null, this.decoder);
            defaultSchemaValues.putIfAbsent(Integer.valueOf(field.pos()), defaultValue);
        }
        return this.data.deepCopy(field.schema(), defaultValue);
    }

    public int hashCode() {
        int result = Arrays.hashCode(this.fieldSetFlags) + 31;
        return (result * 31) + (this.schema == null ? 0 : this.schema.hashCode());
    }

    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj != null && getClass() == obj.getClass()) {
            RecordBuilderBase other = (RecordBuilderBase) obj;
            if (Arrays.equals(this.fieldSetFlags, other.fieldSetFlags)) {
                return this.schema == null ? other.schema == null : this.schema.equals(other.schema);
            }
            return false;
        }
        return false;
    }
}
