package com.flurry.org.apache.avro;

import com.flurry.org.apache.avro.file.DataFileConstants;
import com.flurry.org.codehaus.jackson.JsonFactory;
import com.flurry.org.codehaus.jackson.JsonGenerator;
import com.flurry.org.codehaus.jackson.JsonNode;
import com.flurry.org.codehaus.jackson.JsonParseException;
import com.flurry.org.codehaus.jackson.JsonParser;
import com.flurry.org.codehaus.jackson.map.ObjectMapper;
import com.flurry.org.codehaus.jackson.node.DoubleNode;
import com.prime31.util.IabHelper;
import com.tapjoy.TapjoyConstants;
import com.tapjoy.TapjoyVideoObject;
import com.urbanairship.analytics.EventDataManager;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringReader;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.IdentityHashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

/* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
public abstract class Schema {
    private static final Set<String> FIELD_RESERVED;
    private static final int NO_HASHCODE = Integer.MIN_VALUE;
    static final Map<String, Type> PRIMITIVES;
    private static final Set<String> SCHEMA_RESERVED;
    private static final ThreadLocal<Set> SEEN_EQUALS;
    private static final ThreadLocal<Map> SEEN_HASHCODE;
    private static ThreadLocal<Boolean> validateNames;
    private final Type type;
    static final JsonFactory FACTORY = new JsonFactory();
    static final ObjectMapper MAPPER = new ObjectMapper(FACTORY);
    Props props = new Props(SCHEMA_RESERVED);
    int hashCode = NO_HASHCODE;

    static {
        FACTORY.enable(JsonParser.Feature.ALLOW_COMMENTS);
        FACTORY.setCodec(MAPPER);
        SCHEMA_RESERVED = new HashSet();
        Collections.addAll(SCHEMA_RESERVED, "doc", "fields", "items", TapjoyConstants.TJC_EVENT_IAP_NAME, "namespace", TapjoyConstants.TJC_DISPLAY_AD_SIZE, "symbols", "values", EventDataManager.Events.COLUMN_NAME_TYPE);
        FIELD_RESERVED = new HashSet();
        Collections.addAll(FIELD_RESERVED, "default", "doc", TapjoyConstants.TJC_EVENT_IAP_NAME, "order", EventDataManager.Events.COLUMN_NAME_TYPE);
        SEEN_EQUALS = new ThreadLocal<Set>() { // from class: com.flurry.org.apache.avro.Schema.1
            /* JADX INFO: Access modifiers changed from: protected */
            @Override // java.lang.ThreadLocal
            public Set initialValue() {
                return new HashSet();
            }
        };
        SEEN_HASHCODE = new ThreadLocal<Map>() { // from class: com.flurry.org.apache.avro.Schema.2
            /* JADX INFO: Access modifiers changed from: protected */
            @Override // java.lang.ThreadLocal
            public Map initialValue() {
                return new IdentityHashMap();
            }
        };
        PRIMITIVES = new HashMap();
        PRIMITIVES.put("string", Type.STRING);
        PRIMITIVES.put("bytes", Type.BYTES);
        PRIMITIVES.put("int", Type.INT);
        PRIMITIVES.put("long", Type.LONG);
        PRIMITIVES.put("float", Type.FLOAT);
        PRIMITIVES.put("double", Type.DOUBLE);
        PRIMITIVES.put("boolean", Type.BOOLEAN);
        PRIMITIVES.put(DataFileConstants.NULL_CODEC, Type.NULL);
        validateNames = new ThreadLocal<Boolean>() { // from class: com.flurry.org.apache.avro.Schema.3
            /* JADX INFO: Access modifiers changed from: protected */
            /* JADX WARN: Can't rename method to resolve collision */
            @Override // java.lang.ThreadLocal
            public Boolean initialValue() {
                return true;
            }
        };
    }

    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public enum Type {
        RECORD,
        ENUM,
        ARRAY,
        MAP,
        UNION,
        FIXED,
        STRING,
        BYTES,
        INT,
        LONG,
        FLOAT,
        DOUBLE,
        BOOLEAN,
        NULL;
        
        private String name = name().toLowerCase();

        Type() {
        }

        public String getName() {
            return this.name;
        }
    }

    Schema(Type type) {
        this.type = type;
    }

    /* JADX INFO: Access modifiers changed from: package-private */
    /* renamed from: com.flurry.org.apache.avro.Schema$4  reason: invalid class name */
    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public static /* synthetic */ class AnonymousClass4 {
        static final /* synthetic */ int[] $SwitchMap$org$apache$avro$Schema$Type = new int[Type.values().length];

        static {
            try {
                $SwitchMap$org$apache$avro$Schema$Type[Type.STRING.ordinal()] = 1;
            } catch (NoSuchFieldError e) {
            }
            try {
                $SwitchMap$org$apache$avro$Schema$Type[Type.BYTES.ordinal()] = 2;
            } catch (NoSuchFieldError e2) {
            }
            try {
                $SwitchMap$org$apache$avro$Schema$Type[Type.INT.ordinal()] = 3;
            } catch (NoSuchFieldError e3) {
            }
            try {
                $SwitchMap$org$apache$avro$Schema$Type[Type.LONG.ordinal()] = 4;
            } catch (NoSuchFieldError e4) {
            }
            try {
                $SwitchMap$org$apache$avro$Schema$Type[Type.FLOAT.ordinal()] = 5;
            } catch (NoSuchFieldError e5) {
            }
            try {
                $SwitchMap$org$apache$avro$Schema$Type[Type.DOUBLE.ordinal()] = 6;
            } catch (NoSuchFieldError e6) {
            }
            try {
                $SwitchMap$org$apache$avro$Schema$Type[Type.BOOLEAN.ordinal()] = 7;
            } catch (NoSuchFieldError e7) {
            }
            try {
                $SwitchMap$org$apache$avro$Schema$Type[Type.NULL.ordinal()] = 8;
            } catch (NoSuchFieldError e8) {
            }
            try {
                $SwitchMap$org$apache$avro$Schema$Type[Type.RECORD.ordinal()] = 9;
            } catch (NoSuchFieldError e9) {
            }
            try {
                $SwitchMap$org$apache$avro$Schema$Type[Type.ENUM.ordinal()] = 10;
            } catch (NoSuchFieldError e10) {
            }
            try {
                $SwitchMap$org$apache$avro$Schema$Type[Type.ARRAY.ordinal()] = 11;
            } catch (NoSuchFieldError e11) {
            }
            try {
                $SwitchMap$org$apache$avro$Schema$Type[Type.MAP.ordinal()] = 12;
            } catch (NoSuchFieldError e12) {
            }
            try {
                $SwitchMap$org$apache$avro$Schema$Type[Type.UNION.ordinal()] = 13;
            } catch (NoSuchFieldError e13) {
            }
            try {
                $SwitchMap$org$apache$avro$Schema$Type[Type.FIXED.ordinal()] = 14;
            } catch (NoSuchFieldError e14) {
            }
        }
    }

    public static Schema create(Type type) {
        switch (AnonymousClass4.$SwitchMap$org$apache$avro$Schema$Type[type.ordinal()]) {
            case 1:
                return new StringSchema();
            case 2:
                return new BytesSchema();
            case 3:
                return new IntSchema();
            case 4:
                return new LongSchema();
            case 5:
                return new FloatSchema();
            case 6:
                return new DoubleSchema();
            case IabHelper.BILLING_RESPONSE_RESULT_ITEM_ALREADY_OWNED /* 7 */:
                return new BooleanSchema();
            case 8:
                return new NullSchema();
            default:
                throw new AvroRuntimeException("Can't create a: " + type);
        }
    }

    /* JADX INFO: Access modifiers changed from: package-private */
    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public static final class Props extends LinkedHashMap<String, String> {
        private Set<String> reserved;

        public Props(Set<String> reserved) {
            super(1);
            this.reserved = reserved;
        }

        public void add(String name, String value) {
            if (this.reserved.contains(name)) {
                throw new AvroRuntimeException("Can't set reserved property: " + name);
            }
            if (value == null) {
                throw new AvroRuntimeException("Can't set a property to null: " + name);
            }
            String old = get(name);
            if (old == null) {
                put(name, value);
            } else if (!old.equals(value)) {
                throw new AvroRuntimeException("Can't overwrite property: " + name);
            }
        }

        public void write(JsonGenerator gen) throws IOException {
            for (Map.Entry<String, String> e : entrySet()) {
                gen.writeStringField(e.getKey(), e.getValue());
            }
        }
    }

    public synchronized String getProp(String name) {
        return this.props.get(name);
    }

    public synchronized void addProp(String name, String value) {
        this.props.add(name, value);
        this.hashCode = NO_HASHCODE;
    }

    public Map<String, String> getProps() {
        return Collections.unmodifiableMap(this.props);
    }

    public static Schema createRecord(List<Field> fields) {
        Schema result = createRecord(null, null, null, false);
        result.setFields(fields);
        return result;
    }

    public static Schema createRecord(String name, String doc, String namespace, boolean isError) {
        return new RecordSchema(new Name(name, namespace), doc, isError);
    }

    public static Schema createEnum(String name, String doc, String namespace, List<String> values) {
        return new EnumSchema(new Name(name, namespace), doc, new LockableArrayList(values));
    }

    public static Schema createArray(Schema elementType) {
        return new ArraySchema(elementType);
    }

    public static Schema createMap(Schema valueType) {
        return new MapSchema(valueType);
    }

    public static Schema createUnion(List<Schema> types) {
        return new UnionSchema(new LockableArrayList(types));
    }

    public static Schema createFixed(String name, String doc, String space, int size) {
        return new FixedSchema(new Name(name, space), doc, size);
    }

    public Type getType() {
        return this.type;
    }

    public Field getField(String fieldname) {
        throw new AvroRuntimeException("Not a record: " + this);
    }

    public List<Field> getFields() {
        throw new AvroRuntimeException("Not a record: " + this);
    }

    public void setFields(List<Field> fields) {
        throw new AvroRuntimeException("Not a record: " + this);
    }

    public List<String> getEnumSymbols() {
        throw new AvroRuntimeException("Not an enum: " + this);
    }

    public int getEnumOrdinal(String symbol) {
        throw new AvroRuntimeException("Not an enum: " + this);
    }

    public boolean hasEnumSymbol(String symbol) {
        throw new AvroRuntimeException("Not an enum: " + this);
    }

    public String getName() {
        return this.type.name;
    }

    public String getDoc() {
        return null;
    }

    public String getNamespace() {
        throw new AvroRuntimeException("Not a named type: " + this);
    }

    public String getFullName() {
        return getName();
    }

    public void addAlias(String alias) {
        throw new AvroRuntimeException("Not a named type: " + this);
    }

    public Set<String> getAliases() {
        throw new AvroRuntimeException("Not a named type: " + this);
    }

    public boolean isError() {
        throw new AvroRuntimeException("Not a record: " + this);
    }

    public Schema getElementType() {
        throw new AvroRuntimeException("Not an array: " + this);
    }

    public Schema getValueType() {
        throw new AvroRuntimeException("Not a map: " + this);
    }

    public List<Schema> getTypes() {
        throw new AvroRuntimeException("Not a union: " + this);
    }

    public Integer getIndexNamed(String name) {
        throw new AvroRuntimeException("Not a union: " + this);
    }

    public int getFixedSize() {
        throw new AvroRuntimeException("Not fixed: " + this);
    }

    public String toString() {
        return toString(false);
    }

    public String toString(boolean pretty) {
        try {
            StringWriter writer = new StringWriter();
            JsonGenerator gen = FACTORY.createJsonGenerator(writer);
            if (pretty) {
                gen.useDefaultPrettyPrinter();
            }
            toJson(new Names(), gen);
            gen.flush();
            return writer.toString();
        } catch (IOException e) {
            throw new AvroRuntimeException(e);
        }
    }

    /* JADX INFO: Access modifiers changed from: package-private */
    public void toJson(Names names, JsonGenerator gen) throws IOException {
        if (this.props.size() == 0) {
            gen.writeString(getName());
            return;
        }
        gen.writeStartObject();
        gen.writeStringField(EventDataManager.Events.COLUMN_NAME_TYPE, getName());
        this.props.write(gen);
        gen.writeEndObject();
    }

    /* JADX INFO: Access modifiers changed from: package-private */
    public void fieldsToJson(Names names, JsonGenerator gen) throws IOException {
        throw new AvroRuntimeException("Not a record: " + this);
    }

    public boolean equals(Object o) {
        if (o == this) {
            return true;
        }
        if (o instanceof Schema) {
            Schema that = (Schema) o;
            if (this.type != that.type) {
                return false;
            }
            return equalCachedHash(that) && this.props.equals(that.props);
        }
        return false;
    }

    public final int hashCode() {
        if (this.hashCode == NO_HASHCODE) {
            this.hashCode = computeHash();
        }
        return this.hashCode;
    }

    int computeHash() {
        return getType().hashCode() + this.props.hashCode();
    }

    final boolean equalCachedHash(Schema other) {
        return this.hashCode == other.hashCode || this.hashCode == NO_HASHCODE || other.hashCode == NO_HASHCODE;
    }

    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public static class Field {
        private Set<String> aliases;
        private final JsonNode defaultValue;
        private final String doc;
        private final String name;
        private final Order order;
        private transient int position;
        private final Props props;
        private final Schema schema;

        /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
        public enum Order {
            ASCENDING,
            DESCENDING,
            IGNORE;
            
            private String name = name().toLowerCase();

            Order() {
            }
        }

        public Field(String name, Schema schema, String doc, JsonNode defaultValue) {
            this(name, schema, doc, defaultValue, Order.ASCENDING);
        }

        public Field(String name, Schema schema, String doc, JsonNode defaultValue, Order order) {
            this.position = -1;
            this.props = new Props(Schema.FIELD_RESERVED);
            this.name = Schema.validateName(name);
            this.schema = schema;
            this.doc = doc;
            this.defaultValue = defaultValue;
            this.order = order;
        }

        public String name() {
            return this.name;
        }

        public int pos() {
            return this.position;
        }

        public Schema schema() {
            return this.schema;
        }

        public String doc() {
            return this.doc;
        }

        public JsonNode defaultValue() {
            return this.defaultValue;
        }

        public Order order() {
            return this.order;
        }

        public synchronized String getProp(String name) {
            return this.props.get(name);
        }

        public synchronized void addProp(String name, String value) {
            this.props.add(name, value);
        }

        public Map<String, String> props() {
            return Collections.unmodifiableMap(this.props);
        }

        public void addAlias(String alias) {
            if (this.aliases == null) {
                this.aliases = new LinkedHashSet();
            }
            this.aliases.add(alias);
        }

        public Set<String> aliases() {
            return this.aliases == null ? Collections.emptySet() : Collections.unmodifiableSet(this.aliases);
        }

        public boolean equals(Object other) {
            if (other == this) {
                return true;
            }
            if (other instanceof Field) {
                Field that = (Field) other;
                return this.name.equals(that.name) && this.schema.equals(that.schema) && defaultValueEquals(that.defaultValue) && this.props.equals(that.props);
            }
            return false;
        }

        public int hashCode() {
            return this.name.hashCode() + this.schema.computeHash();
        }

        private boolean defaultValueEquals(JsonNode thatDefaultValue) {
            if (this.defaultValue == null) {
                return thatDefaultValue == null;
            } else if (Double.isNaN(this.defaultValue.getValueAsDouble())) {
                return Double.isNaN(thatDefaultValue.getValueAsDouble());
            } else {
                return this.defaultValue.equals(thatDefaultValue);
            }
        }

        public String toString() {
            return this.name + " type:" + this.schema.type + " pos:" + this.position;
        }
    }

    /* JADX INFO: Access modifiers changed from: private */
    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public static class Name {
        private final String full;
        private final String name;
        private final String space;

        public Name(String name, String space) {
            if (name == null) {
                this.full = null;
                this.space = null;
                this.name = null;
                return;
            }
            int lastDot = name.lastIndexOf(46);
            if (lastDot < 0) {
                this.space = space;
                this.name = Schema.validateName(name);
            } else {
                this.space = name.substring(0, lastDot);
                this.name = Schema.validateName(name.substring(lastDot + 1, name.length()));
            }
            this.full = this.space == null ? this.name : this.space + "." + this.name;
        }

        public boolean equals(Object o) {
            if (o == this) {
                return true;
            }
            if (o instanceof Name) {
                Name that = (Name) o;
                return this.full == null ? that.full == null : this.full.equals(that.full);
            }
            return false;
        }

        public int hashCode() {
            if (this.full == null) {
                return 0;
            }
            return this.full.hashCode();
        }

        public String toString() {
            return this.full;
        }

        public void writeName(Names names, JsonGenerator gen) throws IOException {
            if (this.name != null) {
                gen.writeStringField(TapjoyConstants.TJC_EVENT_IAP_NAME, this.name);
            }
            if (this.space != null) {
                if (!this.space.equals(names.space())) {
                    gen.writeStringField("namespace", this.space);
                }
                if (names.space() == null) {
                    names.space(this.space);
                }
            }
        }

        public String getQualified(String defaultSpace) {
            return (this.space == null || this.space.equals(defaultSpace)) ? this.name : this.full;
        }
    }

    /* JADX INFO: Access modifiers changed from: private */
    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public static abstract class NamedSchema extends Schema {
        Set<Name> aliases;
        final String doc;
        final Name name;

        public NamedSchema(Type type, Name name, String doc) {
            super(type);
            this.name = name;
            this.doc = doc;
            if (PRIMITIVES.containsKey(name.full)) {
                throw new AvroTypeException("Schemas may not be named after primitives: " + name.full);
            }
        }

        @Override // com.flurry.org.apache.avro.Schema
        public String getName() {
            return this.name.name;
        }

        @Override // com.flurry.org.apache.avro.Schema
        public String getDoc() {
            return this.doc;
        }

        @Override // com.flurry.org.apache.avro.Schema
        public String getNamespace() {
            return this.name.space;
        }

        @Override // com.flurry.org.apache.avro.Schema
        public String getFullName() {
            return this.name.full;
        }

        @Override // com.flurry.org.apache.avro.Schema
        public void addAlias(String alias) {
            if (this.aliases == null) {
                this.aliases = new LinkedHashSet();
            }
            this.aliases.add(new Name(alias, this.name.space));
        }

        @Override // com.flurry.org.apache.avro.Schema
        public Set<String> getAliases() {
            Set<String> result = new LinkedHashSet<>();
            if (this.aliases != null) {
                for (Name alias : this.aliases) {
                    result.add(alias.full);
                }
            }
            return result;
        }

        public boolean writeNameRef(Names names, JsonGenerator gen) throws IOException {
            if (!equals(names.get((Object) this.name))) {
                if (this.name.name != null) {
                    names.put(this.name, (Schema) this);
                }
                return false;
            }
            gen.writeString(this.name.getQualified(names.space()));
            return true;
        }

        public void writeName(Names names, JsonGenerator gen) throws IOException {
            this.name.writeName(names, gen);
        }

        public boolean equalNames(NamedSchema that) {
            return this.name.equals(that.name);
        }

        @Override // com.flurry.org.apache.avro.Schema
        int computeHash() {
            return super.computeHash() + this.name.hashCode();
        }

        public void aliasesToJson(JsonGenerator gen) throws IOException {
            if (this.aliases != null && this.aliases.size() != 0) {
                gen.writeFieldName("aliases");
                gen.writeStartArray();
                for (Name alias : this.aliases) {
                    gen.writeString(alias.getQualified(this.name.space));
                }
                gen.writeEndArray();
            }
        }
    }

    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    private static class SeenPair {
        private Object s1;
        private Object s2;

        private SeenPair(Object s1, Object s2) {
            this.s1 = s1;
            this.s2 = s2;
        }

        public boolean equals(Object o) {
            return this.s1 == ((SeenPair) o).s1 && this.s2 == ((SeenPair) o).s2;
        }

        public int hashCode() {
            return System.identityHashCode(this.s1) + System.identityHashCode(this.s2);
        }
    }

    /* JADX INFO: Access modifiers changed from: private */
    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public static class RecordSchema extends NamedSchema {
        private Map<String, Field> fieldMap;
        private List<Field> fields;
        private final boolean isError;

        public RecordSchema(Name name, String doc, boolean isError) {
            super(Type.RECORD, name, doc);
            this.isError = isError;
        }

        @Override // com.flurry.org.apache.avro.Schema
        public boolean isError() {
            return this.isError;
        }

        @Override // com.flurry.org.apache.avro.Schema
        public Field getField(String fieldname) {
            if (this.fieldMap == null) {
                throw new AvroRuntimeException("Schema fields not set yet");
            }
            return this.fieldMap.get(fieldname);
        }

        @Override // com.flurry.org.apache.avro.Schema
        public List<Field> getFields() {
            if (this.fields == null) {
                throw new AvroRuntimeException("Schema fields not set yet");
            }
            return this.fields;
        }

        @Override // com.flurry.org.apache.avro.Schema
        public void setFields(List<Field> fields) {
            if (this.fields != null) {
                throw new AvroRuntimeException("Fields are already set");
            }
            int i = 0;
            this.fieldMap = new HashMap();
            LockableArrayList ff = new LockableArrayList();
            for (Field f : fields) {
                if (f.position == -1) {
                    f.position = i;
                    this.fieldMap.put(f.name(), f);
                    ff.add(f);
                    i++;
                } else {
                    throw new AvroRuntimeException("Field already used: " + f);
                }
            }
            this.fields = ff.lock();
            this.hashCode = Schema.NO_HASHCODE;
        }

        @Override // com.flurry.org.apache.avro.Schema
        public boolean equals(Object o) {
            if (o == this) {
                return true;
            }
            if (o instanceof RecordSchema) {
                RecordSchema that = (RecordSchema) o;
                if (equalCachedHash(that) && equalNames(that) && this.props.equals(that.props)) {
                    Set seen = (Set) Schema.SEEN_EQUALS.get();
                    SeenPair here = new SeenPair(this, o);
                    if (seen.contains(here)) {
                        return true;
                    }
                    boolean first = seen.isEmpty();
                    try {
                        seen.add(here);
                        boolean equals = this.fields.equals(((RecordSchema) o).fields);
                    } finally {
                        if (first) {
                            seen.clear();
                        }
                    }
                }
                return false;
            }
            return false;
        }

        @Override // com.flurry.org.apache.avro.Schema.NamedSchema, com.flurry.org.apache.avro.Schema
        int computeHash() {
            Map seen = (Map) Schema.SEEN_HASHCODE.get();
            if (seen.containsKey(this)) {
                return 0;
            }
            boolean first = seen.isEmpty();
            try {
                seen.put(this, this);
                int computeHash = super.computeHash() + this.fields.hashCode();
            } finally {
                if (first) {
                    seen.clear();
                }
            }
        }

        @Override // com.flurry.org.apache.avro.Schema
        void toJson(Names names, JsonGenerator gen) throws IOException {
            if (writeNameRef(names, gen)) {
                return;
            }
            String savedSpace = names.space;
            gen.writeStartObject();
            gen.writeStringField(EventDataManager.Events.COLUMN_NAME_TYPE, this.isError ? "error" : "record");
            writeName(names, gen);
            names.space = this.name.space;
            if (getDoc() != null) {
                gen.writeStringField("doc", getDoc());
            }
            gen.writeFieldName("fields");
            fieldsToJson(names, gen);
            this.props.write(gen);
            aliasesToJson(gen);
            gen.writeEndObject();
            names.space = savedSpace;
        }

        @Override // com.flurry.org.apache.avro.Schema
        void fieldsToJson(Names names, JsonGenerator gen) throws IOException {
            gen.writeStartArray();
            for (Field f : this.fields) {
                gen.writeStartObject();
                gen.writeStringField(TapjoyConstants.TJC_EVENT_IAP_NAME, f.name());
                gen.writeFieldName(EventDataManager.Events.COLUMN_NAME_TYPE);
                f.schema().toJson(names, gen);
                if (f.doc() != null) {
                    gen.writeStringField("doc", f.doc());
                }
                if (f.defaultValue() != null) {
                    gen.writeFieldName("default");
                    gen.writeTree(f.defaultValue());
                }
                if (f.order() != Field.Order.ASCENDING) {
                    gen.writeStringField("order", f.order().name);
                }
                if (f.aliases != null && f.aliases.size() != 0) {
                    gen.writeFieldName("aliases");
                    gen.writeStartArray();
                    for (String alias : f.aliases) {
                        gen.writeString(alias);
                    }
                    gen.writeEndArray();
                }
                f.props.write(gen);
                gen.writeEndObject();
            }
            gen.writeEndArray();
        }
    }

    /* JADX INFO: Access modifiers changed from: private */
    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public static class EnumSchema extends NamedSchema {
        private final Map<String, Integer> ordinals;
        private final List<String> symbols;

        public EnumSchema(Name name, String doc, LockableArrayList<String> symbols) {
            super(Type.ENUM, name, doc);
            this.symbols = symbols.lock();
            this.ordinals = new HashMap();
            int i = 0;
            Iterator i$ = symbols.iterator();
            while (i$.hasNext()) {
                String symbol = i$.next();
                int i2 = i + 1;
                if (this.ordinals.put(Schema.validateName(symbol), Integer.valueOf(i)) != null) {
                    throw new SchemaParseException("Duplicate enum symbol: " + symbol);
                }
                i = i2;
            }
        }

        @Override // com.flurry.org.apache.avro.Schema
        public List<String> getEnumSymbols() {
            return this.symbols;
        }

        @Override // com.flurry.org.apache.avro.Schema
        public boolean hasEnumSymbol(String symbol) {
            return this.ordinals.containsKey(symbol);
        }

        @Override // com.flurry.org.apache.avro.Schema
        public int getEnumOrdinal(String symbol) {
            return this.ordinals.get(symbol).intValue();
        }

        @Override // com.flurry.org.apache.avro.Schema
        public boolean equals(Object o) {
            if (o == this) {
                return true;
            }
            if (o instanceof EnumSchema) {
                EnumSchema that = (EnumSchema) o;
                return equalCachedHash(that) && equalNames(that) && this.symbols.equals(that.symbols) && this.props.equals(that.props);
            }
            return false;
        }

        @Override // com.flurry.org.apache.avro.Schema.NamedSchema, com.flurry.org.apache.avro.Schema
        int computeHash() {
            return super.computeHash() + this.symbols.hashCode();
        }

        @Override // com.flurry.org.apache.avro.Schema
        void toJson(Names names, JsonGenerator gen) throws IOException {
            if (!writeNameRef(names, gen)) {
                gen.writeStartObject();
                gen.writeStringField(EventDataManager.Events.COLUMN_NAME_TYPE, "enum");
                writeName(names, gen);
                if (getDoc() != null) {
                    gen.writeStringField("doc", getDoc());
                }
                gen.writeArrayFieldStart("symbols");
                for (String symbol : this.symbols) {
                    gen.writeString(symbol);
                }
                gen.writeEndArray();
                this.props.write(gen);
                aliasesToJson(gen);
                gen.writeEndObject();
            }
        }
    }

    /* JADX INFO: Access modifiers changed from: private */
    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public static class ArraySchema extends Schema {
        private final Schema elementType;

        public ArraySchema(Schema elementType) {
            super(Type.ARRAY);
            this.elementType = elementType;
        }

        @Override // com.flurry.org.apache.avro.Schema
        public Schema getElementType() {
            return this.elementType;
        }

        @Override // com.flurry.org.apache.avro.Schema
        public boolean equals(Object o) {
            if (o == this) {
                return true;
            }
            if (o instanceof ArraySchema) {
                ArraySchema that = (ArraySchema) o;
                return equalCachedHash(that) && this.elementType.equals(that.elementType) && this.props.equals(that.props);
            }
            return false;
        }

        @Override // com.flurry.org.apache.avro.Schema
        int computeHash() {
            return super.computeHash() + this.elementType.computeHash();
        }

        @Override // com.flurry.org.apache.avro.Schema
        void toJson(Names names, JsonGenerator gen) throws IOException {
            gen.writeStartObject();
            gen.writeStringField(EventDataManager.Events.COLUMN_NAME_TYPE, "array");
            gen.writeFieldName("items");
            this.elementType.toJson(names, gen);
            this.props.write(gen);
            gen.writeEndObject();
        }
    }

    /* JADX INFO: Access modifiers changed from: private */
    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public static class MapSchema extends Schema {
        private final Schema valueType;

        public MapSchema(Schema valueType) {
            super(Type.MAP);
            this.valueType = valueType;
        }

        @Override // com.flurry.org.apache.avro.Schema
        public Schema getValueType() {
            return this.valueType;
        }

        @Override // com.flurry.org.apache.avro.Schema
        public boolean equals(Object o) {
            if (o == this) {
                return true;
            }
            if (o instanceof MapSchema) {
                MapSchema that = (MapSchema) o;
                return equalCachedHash(that) && this.valueType.equals(that.valueType) && this.props.equals(that.props);
            }
            return false;
        }

        @Override // com.flurry.org.apache.avro.Schema
        int computeHash() {
            return super.computeHash() + this.valueType.computeHash();
        }

        @Override // com.flurry.org.apache.avro.Schema
        void toJson(Names names, JsonGenerator gen) throws IOException {
            gen.writeStartObject();
            gen.writeStringField(EventDataManager.Events.COLUMN_NAME_TYPE, "map");
            gen.writeFieldName("values");
            this.valueType.toJson(names, gen);
            this.props.write(gen);
            gen.writeEndObject();
        }
    }

    /* JADX INFO: Access modifiers changed from: private */
    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public static class UnionSchema extends Schema {
        private final Map<String, Integer> indexByName;
        private final List<Schema> types;

        public UnionSchema(LockableArrayList<Schema> types) {
            super(Type.UNION);
            this.indexByName = new HashMap();
            this.types = types.lock();
            int index = 0;
            Iterator i$ = types.iterator();
            while (i$.hasNext()) {
                Schema type = i$.next();
                if (type.getType() == Type.UNION) {
                    throw new AvroRuntimeException("Nested union: " + this);
                }
                String name = type.getFullName();
                if (name == null) {
                    throw new AvroRuntimeException("Nameless in union:" + this);
                }
                int index2 = index + 1;
                if (this.indexByName.put(name, Integer.valueOf(index)) != null) {
                    throw new AvroRuntimeException("Duplicate in union:" + name);
                }
                index = index2;
            }
        }

        @Override // com.flurry.org.apache.avro.Schema
        public List<Schema> getTypes() {
            return this.types;
        }

        @Override // com.flurry.org.apache.avro.Schema
        public Integer getIndexNamed(String name) {
            return this.indexByName.get(name);
        }

        @Override // com.flurry.org.apache.avro.Schema
        public boolean equals(Object o) {
            if (o == this) {
                return true;
            }
            if (o instanceof UnionSchema) {
                UnionSchema that = (UnionSchema) o;
                return equalCachedHash(that) && this.types.equals(that.types) && this.props.equals(that.props);
            }
            return false;
        }

        @Override // com.flurry.org.apache.avro.Schema
        int computeHash() {
            int hash = super.computeHash();
            for (Schema type : this.types) {
                hash += type.computeHash();
            }
            return hash;
        }

        @Override // com.flurry.org.apache.avro.Schema
        public void addProp(String name, String value) {
            throw new AvroRuntimeException("Can't set properties on a union: " + this);
        }

        @Override // com.flurry.org.apache.avro.Schema
        void toJson(Names names, JsonGenerator gen) throws IOException {
            gen.writeStartArray();
            for (Schema type : this.types) {
                type.toJson(names, gen);
            }
            gen.writeEndArray();
        }
    }

    /* JADX INFO: Access modifiers changed from: private */
    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public static class FixedSchema extends NamedSchema {
        private final int size;

        public FixedSchema(Name name, String doc, int size) {
            super(Type.FIXED, name, doc);
            if (size < 0) {
                throw new IllegalArgumentException("Invalid fixed size: " + size);
            }
            this.size = size;
        }

        @Override // com.flurry.org.apache.avro.Schema
        public int getFixedSize() {
            return this.size;
        }

        @Override // com.flurry.org.apache.avro.Schema
        public boolean equals(Object o) {
            if (o == this) {
                return true;
            }
            if (o instanceof FixedSchema) {
                FixedSchema that = (FixedSchema) o;
                return equalCachedHash(that) && equalNames(that) && this.size == that.size && this.props.equals(that.props);
            }
            return false;
        }

        @Override // com.flurry.org.apache.avro.Schema.NamedSchema, com.flurry.org.apache.avro.Schema
        int computeHash() {
            return super.computeHash() + this.size;
        }

        @Override // com.flurry.org.apache.avro.Schema
        void toJson(Names names, JsonGenerator gen) throws IOException {
            if (!writeNameRef(names, gen)) {
                gen.writeStartObject();
                gen.writeStringField(EventDataManager.Events.COLUMN_NAME_TYPE, "fixed");
                writeName(names, gen);
                if (getDoc() != null) {
                    gen.writeStringField("doc", getDoc());
                }
                gen.writeNumberField(TapjoyConstants.TJC_DISPLAY_AD_SIZE, this.size);
                this.props.write(gen);
                aliasesToJson(gen);
                gen.writeEndObject();
            }
        }
    }

    /* JADX INFO: Access modifiers changed from: private */
    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public static class StringSchema extends Schema {
        public StringSchema() {
            super(Type.STRING);
        }
    }

    /* JADX INFO: Access modifiers changed from: private */
    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public static class BytesSchema extends Schema {
        public BytesSchema() {
            super(Type.BYTES);
        }
    }

    /* JADX INFO: Access modifiers changed from: private */
    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public static class IntSchema extends Schema {
        public IntSchema() {
            super(Type.INT);
        }
    }

    /* JADX INFO: Access modifiers changed from: private */
    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public static class LongSchema extends Schema {
        public LongSchema() {
            super(Type.LONG);
        }
    }

    /* JADX INFO: Access modifiers changed from: private */
    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public static class FloatSchema extends Schema {
        public FloatSchema() {
            super(Type.FLOAT);
        }
    }

    /* JADX INFO: Access modifiers changed from: private */
    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public static class DoubleSchema extends Schema {
        public DoubleSchema() {
            super(Type.DOUBLE);
        }
    }

    /* JADX INFO: Access modifiers changed from: private */
    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public static class BooleanSchema extends Schema {
        public BooleanSchema() {
            super(Type.BOOLEAN);
        }
    }

    /* JADX INFO: Access modifiers changed from: private */
    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public static class NullSchema extends Schema {
        public NullSchema() {
            super(Type.NULL);
        }
    }

    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public static class Parser {
        private Names names = new Names();
        private boolean validate = true;

        public Parser addTypes(Map<String, Schema> types) {
            for (Schema s : types.values()) {
                this.names.add(s);
            }
            return this;
        }

        public Map<String, Schema> getTypes() {
            Map<String, Schema> result = new LinkedHashMap<>();
            for (Schema s : this.names.values()) {
                result.put(s.getFullName(), s);
            }
            return result;
        }

        public Parser setValidate(boolean validate) {
            this.validate = validate;
            return this;
        }

        public boolean getValidate() {
            return this.validate;
        }

        public Schema parse(File file) throws IOException {
            return parse(Schema.FACTORY.createJsonParser(file));
        }

        public Schema parse(InputStream in) throws IOException {
            return parse(Schema.FACTORY.createJsonParser(in));
        }

        public Schema parse(String s) {
            try {
                return parse(Schema.FACTORY.createJsonParser(new StringReader(s)));
            } catch (IOException e) {
                throw new SchemaParseException(e);
            }
        }

        private Schema parse(JsonParser parser) throws IOException {
            boolean saved = ((Boolean) Schema.validateNames.get()).booleanValue();
            try {
                try {
                    Schema.validateNames.set(Boolean.valueOf(this.validate));
                    return Schema.parse(Schema.MAPPER.readTree(parser), this.names);
                } catch (JsonParseException e) {
                    throw new SchemaParseException(e);
                }
            } finally {
                Schema.validateNames.set(Boolean.valueOf(saved));
            }
        }
    }

    public static Schema parse(File file) throws IOException {
        return new Parser().parse(file);
    }

    public static Schema parse(InputStream in) throws IOException {
        return new Parser().parse(in);
    }

    public static Schema parse(String jsonSchema) {
        return new Parser().parse(jsonSchema);
    }

    public static Schema parse(String jsonSchema, boolean validate) {
        return new Parser().setValidate(validate).parse(jsonSchema);
    }

    /* JADX INFO: Access modifiers changed from: package-private */
    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public static class Names extends LinkedHashMap<Name, Schema> {
        private String space;

        public Names() {
        }

        public Names(String space) {
            this.space = space;
        }

        public String space() {
            return this.space;
        }

        public void space(String space) {
            this.space = space;
        }

        @Override // java.util.LinkedHashMap, java.util.HashMap, java.util.AbstractMap, java.util.Map
        public Schema get(Object o) {
            Name name;
            if (o instanceof String) {
                Type primitive = Schema.PRIMITIVES.get((String) o);
                if (primitive != null) {
                    return Schema.create(primitive);
                }
                name = new Name((String) o, this.space);
            } else {
                name = (Name) o;
            }
            return (Schema) super.get((Object) name);
        }

        public boolean contains(Schema schema) {
            return get((Object) ((NamedSchema) schema).name) != null;
        }

        public void add(Schema schema) {
            put(((NamedSchema) schema).name, schema);
        }

        @Override // java.util.HashMap, java.util.AbstractMap, java.util.Map
        public Schema put(Name name, Schema schema) {
            if (containsKey(name)) {
                throw new SchemaParseException("Can't redefine: " + name);
            }
            return (Schema) super.put((Names) name, (Name) schema);
        }
    }

    /* JADX INFO: Access modifiers changed from: private */
    public static String validateName(String name) {
        if (validateNames.get().booleanValue()) {
            int length = name.length();
            if (length == 0) {
                throw new SchemaParseException("Empty name");
            }
            char first = name.charAt(0);
            if (!Character.isLetter(first) && first != '_') {
                throw new SchemaParseException("Illegal initial character: " + name);
            }
            for (int i = 1; i < length; i++) {
                char c = name.charAt(i);
                if (!Character.isLetterOrDigit(c) && c != '_') {
                    throw new SchemaParseException("Illegal character in: " + name);
                }
            }
        }
        return name;
    }

    /* JADX INFO: Access modifiers changed from: package-private */
    public static Schema parse(JsonNode schema, Names names) {
        Schema result;
        Set<String> aliases;
        if (schema.isTextual()) {
            Schema result2 = names.get((Object) schema.getTextValue());
            if (result2 == null) {
                throw new SchemaParseException("Undefined name: " + schema);
            }
            return result2;
        } else if (schema.isObject()) {
            String type = getRequiredText(schema, EventDataManager.Events.COLUMN_NAME_TYPE, "No type");
            Name name = null;
            String savedSpace = null;
            String doc = null;
            if (type.equals("record") || type.equals("error") || type.equals("enum") || type.equals("fixed")) {
                String space = getOptionalText(schema, "namespace");
                doc = getOptionalText(schema, "doc");
                if (space == null) {
                    space = names.space();
                }
                name = new Name(getRequiredText(schema, TapjoyConstants.TJC_EVENT_IAP_NAME, "No name in schema"), space);
                if (name.space != null) {
                    savedSpace = names.space();
                    names.space(name.space);
                }
            }
            if (PRIMITIVES.containsKey(type)) {
                result = create(PRIMITIVES.get(type));
            } else if (type.equals("record") || type.equals("error")) {
                List<Field> fields = new ArrayList<>();
                result = new RecordSchema(name, doc, type.equals("error"));
                if (name != null) {
                    names.add(result);
                }
                JsonNode fieldsNode = schema.get("fields");
                if (fieldsNode == null || !fieldsNode.isArray()) {
                    throw new SchemaParseException("Record has no fields: " + schema);
                }
                Iterator i$ = fieldsNode.iterator();
                while (i$.hasNext()) {
                    JsonNode field = i$.next();
                    String fieldName = getRequiredText(field, TapjoyConstants.TJC_EVENT_IAP_NAME, "No field name");
                    String fieldDoc = getOptionalText(field, "doc");
                    JsonNode fieldTypeNode = field.get(EventDataManager.Events.COLUMN_NAME_TYPE);
                    if (fieldTypeNode == null) {
                        throw new SchemaParseException("No field type: " + field);
                    }
                    if (fieldTypeNode.isTextual() && names.get((Object) fieldTypeNode.getTextValue()) == null) {
                        throw new SchemaParseException(fieldTypeNode + " is not a defined name. The type of the \"" + fieldName + "\" field must be a defined name or a {\"type\": ...} expression.");
                    }
                    Schema fieldSchema = parse(fieldTypeNode, names);
                    Field.Order order = Field.Order.ASCENDING;
                    JsonNode orderNode = field.get("order");
                    if (orderNode != null) {
                        order = Field.Order.valueOf(orderNode.getTextValue().toUpperCase());
                    }
                    JsonNode defaultValue = field.get("default");
                    if (defaultValue != null && ((Type.FLOAT.equals(fieldSchema.getType()) || Type.DOUBLE.equals(fieldSchema.getType())) && defaultValue.isTextual())) {
                        defaultValue = new DoubleNode(Double.valueOf(defaultValue.getTextValue()).doubleValue());
                    }
                    Field f = new Field(fieldName, fieldSchema, fieldDoc, defaultValue, order);
                    Iterator<String> i = field.getFieldNames();
                    while (i.hasNext()) {
                        String prop = i.next();
                        String value = field.get(prop).getTextValue();
                        if (!FIELD_RESERVED.contains(prop) && value != null) {
                            f.addProp(prop, value);
                        }
                    }
                    f.aliases = parseAliases(field);
                    fields.add(f);
                }
                result.setFields(fields);
            } else if (type.equals("enum")) {
                JsonNode symbolsNode = schema.get("symbols");
                if (symbolsNode == null || !symbolsNode.isArray()) {
                    throw new SchemaParseException("Enum has no symbols: " + schema);
                }
                LockableArrayList<String> symbols = new LockableArrayList<>();
                Iterator i$2 = symbolsNode.iterator();
                while (i$2.hasNext()) {
                    JsonNode n = i$2.next();
                    symbols.add(n.getTextValue());
                }
                result = new EnumSchema(name, doc, symbols);
                if (name != null) {
                    names.add(result);
                }
            } else if (type.equals("array")) {
                JsonNode itemsNode = schema.get("items");
                if (itemsNode == null) {
                    throw new SchemaParseException("Array has no items type: " + schema);
                }
                result = new ArraySchema(parse(itemsNode, names));
            } else if (type.equals("map")) {
                JsonNode valuesNode = schema.get("values");
                if (valuesNode == null) {
                    throw new SchemaParseException("Map has no values type: " + schema);
                }
                result = new MapSchema(parse(valuesNode, names));
            } else if (type.equals("fixed")) {
                JsonNode sizeNode = schema.get(TapjoyConstants.TJC_DISPLAY_AD_SIZE);
                if (sizeNode == null || !sizeNode.isInt()) {
                    throw new SchemaParseException("Invalid or no size: " + schema);
                }
                result = new FixedSchema(name, doc, sizeNode.getIntValue());
                if (name != null) {
                    names.add(result);
                }
            } else {
                throw new SchemaParseException("Type not supported: " + type);
            }
            Iterator<String> i2 = schema.getFieldNames();
            while (i2.hasNext()) {
                String prop2 = i2.next();
                String value2 = schema.get(prop2).getTextValue();
                if (!SCHEMA_RESERVED.contains(prop2) && value2 != null) {
                    result.addProp(prop2, value2);
                }
            }
            if (savedSpace != null) {
                names.space(savedSpace);
            }
            if (!(result instanceof NamedSchema) || (aliases = parseAliases(schema)) == null) {
                return result;
            }
            for (String alias : aliases) {
                result.addAlias(alias);
            }
            return result;
        } else if (schema.isArray()) {
            LockableArrayList<Schema> types = new LockableArrayList<>(schema.size());
            Iterator i$3 = schema.iterator();
            while (i$3.hasNext()) {
                JsonNode typeNode = i$3.next();
                types.add(parse(typeNode, names));
            }
            return new UnionSchema(types);
        } else {
            throw new SchemaParseException("Schema not yet supported: " + schema);
        }
    }

    private static Set<String> parseAliases(JsonNode node) {
        JsonNode aliasesNode = node.get("aliases");
        if (aliasesNode == null) {
            return null;
        }
        if (!aliasesNode.isArray()) {
            throw new SchemaParseException("aliases not an array: " + node);
        }
        Set<String> aliases = new LinkedHashSet<>();
        Iterator i$ = aliasesNode.iterator();
        while (i$.hasNext()) {
            JsonNode aliasNode = i$.next();
            if (!aliasNode.isTextual()) {
                throw new SchemaParseException("alias not a string: " + aliasNode);
            }
            aliases.add(aliasNode.getTextValue());
        }
        return aliases;
    }

    private static String getRequiredText(JsonNode container, String key, String error) {
        String out = getOptionalText(container, key);
        if (out == null) {
            throw new SchemaParseException(error + ": " + container);
        }
        return out;
    }

    private static String getOptionalText(JsonNode container, String key) {
        JsonNode jsonNode = container.get(key);
        if (jsonNode != null) {
            return jsonNode.getTextValue();
        }
        return null;
    }

    static JsonNode parseJson(String s) {
        try {
            return MAPPER.readTree(FACTORY.createJsonParser(new StringReader(s)));
        } catch (JsonParseException e) {
            throw new RuntimeException(e);
        } catch (IOException e2) {
            throw new RuntimeException(e2);
        }
    }

    public static Schema applyAliases(Schema writer, Schema reader) {
        if (writer != reader) {
            Map<Schema, Schema> seen = new IdentityHashMap<>(1);
            Map<Name, Name> aliases = new HashMap<>(1);
            Map<Name, Map<String, String>> fieldAliases = new HashMap<>(1);
            getAliases(reader, seen, aliases, fieldAliases);
            if (aliases.size() != 0 || fieldAliases.size() != 0) {
                seen.clear();
                return applyAliases(writer, seen, aliases, fieldAliases);
            }
            return writer;
        }
        return writer;
    }

    private static Schema applyAliases(Schema s, Map<Schema, Schema> seen, Map<Name, Name> aliases, Map<Name, Map<String, String>> fieldAliases) {
        Name name = s instanceof NamedSchema ? ((NamedSchema) s).name : null;
        Schema result = s;
        switch (AnonymousClass4.$SwitchMap$org$apache$avro$Schema$Type[s.getType().ordinal()]) {
            case 9:
                if (!seen.containsKey(s)) {
                    if (aliases.containsKey(name)) {
                        name = aliases.get(name);
                    }
                    result = createRecord(name.full, s.getDoc(), null, s.isError());
                    seen.put(s, result);
                    List<Field> newFields = new ArrayList<>();
                    for (Field f : s.getFields()) {
                        Schema fSchema = applyAliases(f.schema, seen, aliases, fieldAliases);
                        String fName = getFieldAlias(name, f.name, fieldAliases);
                        Field newF = new Field(fName, fSchema, f.doc, f.defaultValue, f.order);
                        newF.props.putAll(f.props);
                        newFields.add(newF);
                    }
                    result.setFields(newFields);
                    break;
                } else {
                    return seen.get(s);
                }
            case TapjoyVideoObject.BUTTON_MAX /* 10 */:
                if (aliases.containsKey(name)) {
                    result = createEnum(aliases.get(name).full, s.getDoc(), null, s.getEnumSymbols());
                    break;
                }
                break;
            case 11:
                Schema e = applyAliases(s.getElementType(), seen, aliases, fieldAliases);
                if (e != s.getElementType()) {
                    result = createArray(e);
                    break;
                }
                break;
            case 12:
                Schema v = applyAliases(s.getValueType(), seen, aliases, fieldAliases);
                if (v != s.getValueType()) {
                    result = createMap(v);
                    break;
                }
                break;
            case 13:
                List<Schema> types = new ArrayList<>();
                for (Schema branch : s.getTypes()) {
                    types.add(applyAliases(branch, seen, aliases, fieldAliases));
                }
                result = createUnion(types);
                break;
            case 14:
                if (aliases.containsKey(name)) {
                    result = createFixed(aliases.get(name).full, s.getDoc(), null, s.getFixedSize());
                    break;
                }
                break;
        }
        if (result != s) {
            result.props.putAll(s.props);
        }
        return result;
    }

    private static void getAliases(Schema schema, Map<Schema, Schema> seen, Map<Name, Name> aliases, Map<Name, Map<String, String>> fieldAliases) {
        if (schema instanceof NamedSchema) {
            NamedSchema namedSchema = (NamedSchema) schema;
            if (namedSchema.aliases != null) {
                for (Name alias : namedSchema.aliases) {
                    aliases.put(alias, namedSchema.name);
                }
            }
        }
        switch (AnonymousClass4.$SwitchMap$org$apache$avro$Schema$Type[schema.getType().ordinal()]) {
            case 9:
                if (!seen.containsKey(schema)) {
                    seen.put(schema, schema);
                    RecordSchema record = (RecordSchema) schema;
                    for (Field field : schema.getFields()) {
                        if (field.aliases != null) {
                            for (String fieldAlias : field.aliases) {
                                Map<String, String> recordAliases = fieldAliases.get(record.name);
                                if (recordAliases == null) {
                                    Name name = record.name;
                                    recordAliases = new HashMap<>();
                                    fieldAliases.put(name, recordAliases);
                                }
                                recordAliases.put(fieldAlias, field.name);
                            }
                        }
                        getAliases(field.schema, seen, aliases, fieldAliases);
                    }
                    if (record.aliases != null && fieldAliases.containsKey(record.name)) {
                        for (Name recordAlias : record.aliases) {
                            fieldAliases.put(recordAlias, fieldAliases.get(record.name));
                        }
                        return;
                    }
                    return;
                }
                return;
            case TapjoyVideoObject.BUTTON_MAX /* 10 */:
            default:
                return;
            case 11:
                getAliases(schema.getElementType(), seen, aliases, fieldAliases);
                return;
            case 12:
                getAliases(schema.getValueType(), seen, aliases, fieldAliases);
                return;
            case 13:
                for (Schema s : schema.getTypes()) {
                    getAliases(s, seen, aliases, fieldAliases);
                }
                return;
        }
    }

    private static String getFieldAlias(Name record, String field, Map<Name, Map<String, String>> fieldAliases) {
        String alias;
        Map<String, String> recordAliases = fieldAliases.get(record);
        return (recordAliases == null || (alias = recordAliases.get(field)) == null) ? field : alias;
    }

    /* JADX INFO: Access modifiers changed from: package-private */
    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public static class LockableArrayList<E> extends ArrayList<E> {
        private static final long serialVersionUID = 1;
        private boolean locked;

        public LockableArrayList() {
            this.locked = false;
        }

        public LockableArrayList(int size) {
            super(size);
            this.locked = false;
        }

        public LockableArrayList(List<E> types) {
            super(types);
            this.locked = false;
        }

        public List<E> lock() {
            this.locked = true;
            return this;
        }

        private void ensureUnlocked() {
            if (this.locked) {
                throw new IllegalStateException();
            }
        }

        @Override // java.util.ArrayList, java.util.AbstractList, java.util.AbstractCollection, java.util.Collection, java.util.List
        public boolean add(E e) {
            ensureUnlocked();
            return super.add(e);
        }

        @Override // java.util.ArrayList, java.util.AbstractCollection, java.util.Collection, java.util.List
        public boolean remove(Object o) {
            ensureUnlocked();
            return super.remove(o);
        }

        @Override // java.util.ArrayList, java.util.AbstractList, java.util.List
        public E remove(int index) {
            ensureUnlocked();
            return (E) super.remove(index);
        }

        @Override // java.util.ArrayList, java.util.AbstractCollection, java.util.Collection, java.util.List
        public boolean addAll(Collection<? extends E> c) {
            ensureUnlocked();
            return super.addAll(c);
        }

        @Override // java.util.ArrayList, java.util.AbstractList, java.util.List
        public boolean addAll(int index, Collection<? extends E> c) {
            ensureUnlocked();
            return super.addAll(index, c);
        }

        @Override // java.util.ArrayList, java.util.AbstractCollection, java.util.Collection, java.util.List
        public boolean removeAll(Collection<?> c) {
            ensureUnlocked();
            return super.removeAll(c);
        }

        @Override // java.util.ArrayList, java.util.AbstractCollection, java.util.Collection, java.util.List
        public boolean retainAll(Collection<?> c) {
            ensureUnlocked();
            return super.retainAll(c);
        }

        @Override // java.util.ArrayList, java.util.AbstractList, java.util.AbstractCollection, java.util.Collection, java.util.List
        public void clear() {
            ensureUnlocked();
            super.clear();
        }
    }
}
