package com.flurry.org.apache.avro;

import com.flurry.org.apache.avro.Schema;
import com.flurry.org.apache.avro.file.DataFileConstants;
import com.flurry.org.codehaus.jackson.JsonGenerator;
import com.flurry.org.codehaus.jackson.JsonNode;
import com.flurry.org.codehaus.jackson.JsonParser;
import com.tapjoy.TapjoyConstants;
import com.urbanairship.analytics.EventDataManager;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringWriter;
import java.security.MessageDigest;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

/* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
public class Protocol {
    private static final Set<String> MESSAGE_RESERVED = new HashSet();
    private static final Set<String> PROTOCOL_RESERVED;
    public static final Schema SYSTEM_ERROR;
    public static final Schema SYSTEM_ERRORS;
    public static final long VERSION = 1;
    private String doc;
    private byte[] md5;
    private Map<String, Message> messages;
    private String name;
    private String namespace;
    Schema.Props props;
    private Schema.Names types;

    static {
        Collections.addAll(MESSAGE_RESERVED, "doc", "response", "request", "errors", "one-way");
        SYSTEM_ERROR = Schema.create(Schema.Type.STRING);
        List<Schema> errors = new ArrayList<>();
        errors.add(SYSTEM_ERROR);
        SYSTEM_ERRORS = Schema.createUnion(errors);
        PROTOCOL_RESERVED = new HashSet();
        Collections.addAll(PROTOCOL_RESERVED, "namespace", "protocol", "doc", "messages", "types", "errors");
    }

    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public class Message {
        private String doc;
        private String name;
        private final Schema.Props props;
        private Schema request;

        private Message(String name, String doc, Map<String, String> propMap, Schema request) {
            this.props = new Schema.Props(Protocol.MESSAGE_RESERVED);
            this.name = name;
            this.doc = doc;
            this.request = request;
            if (propMap != null) {
                for (Map.Entry<String, String> prop : propMap.entrySet()) {
                    addProp(prop.getKey(), prop.getValue());
                }
            }
        }

        public String getName() {
            return this.name;
        }

        public Schema getRequest() {
            return this.request;
        }

        public Schema getResponse() {
            return Schema.create(Schema.Type.NULL);
        }

        public Schema getErrors() {
            return Schema.createUnion(new ArrayList());
        }

        public boolean isOneWay() {
            return true;
        }

        public synchronized String getProp(String name) {
            return this.props.get(name);
        }

        public synchronized void addProp(String name, String value) {
            this.props.add(name, value);
        }

        public Map<String, String> getProps() {
            return Collections.unmodifiableMap(this.props);
        }

        public String toString() {
            try {
                StringWriter writer = new StringWriter();
                JsonGenerator gen = Schema.FACTORY.createJsonGenerator(writer);
                toJson(gen);
                gen.flush();
                return writer.toString();
            } catch (IOException e) {
                throw new AvroRuntimeException(e);
            }
        }

        void toJson(JsonGenerator gen) throws IOException {
            gen.writeStartObject();
            if (this.doc != null) {
                gen.writeStringField("doc", this.doc);
            }
            this.props.write(gen);
            gen.writeFieldName("request");
            this.request.fieldsToJson(Protocol.this.types, gen);
            toJson1(gen);
            gen.writeEndObject();
        }

        void toJson1(JsonGenerator gen) throws IOException {
            gen.writeStringField("response", DataFileConstants.NULL_CODEC);
            gen.writeBooleanField("one-way", true);
        }

        public boolean equals(Object o) {
            if (o == this) {
                return true;
            }
            if (o instanceof Message) {
                Message that = (Message) o;
                return this.name.equals(that.name) && this.request.equals(that.request) && this.props.equals(that.props);
            }
            return false;
        }

        public int hashCode() {
            return this.name.hashCode() + this.request.hashCode() + this.props.hashCode();
        }

        public String getDoc() {
            return this.doc;
        }
    }

    /* JADX INFO: Access modifiers changed from: private */
    /* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
    public class TwoWayMessage extends Message {
        private Schema errors;
        private Schema response;

        private TwoWayMessage(String name, String doc, Map<String, String> propMap, Schema request, Schema response, Schema errors) {
            super(name, doc, propMap, request);
            this.response = response;
            this.errors = errors;
        }

        @Override // com.flurry.org.apache.avro.Protocol.Message
        public Schema getResponse() {
            return this.response;
        }

        @Override // com.flurry.org.apache.avro.Protocol.Message
        public Schema getErrors() {
            return this.errors;
        }

        @Override // com.flurry.org.apache.avro.Protocol.Message
        public boolean isOneWay() {
            return false;
        }

        @Override // com.flurry.org.apache.avro.Protocol.Message
        public boolean equals(Object o) {
            if (super.equals(o) && (o instanceof TwoWayMessage)) {
                TwoWayMessage that = (TwoWayMessage) o;
                return this.response.equals(that.response) && this.errors.equals(that.errors);
            }
            return false;
        }

        @Override // com.flurry.org.apache.avro.Protocol.Message
        public int hashCode() {
            return super.hashCode() + this.response.hashCode() + this.errors.hashCode();
        }

        @Override // com.flurry.org.apache.avro.Protocol.Message
        void toJson1(JsonGenerator gen) throws IOException {
            gen.writeFieldName("response");
            this.response.toJson(Protocol.this.types, gen);
            List<Schema> errs = this.errors.getTypes();
            if (errs.size() > 1) {
                Schema union = Schema.createUnion(errs.subList(1, errs.size()));
                gen.writeFieldName("errors");
                union.toJson(Protocol.this.types, gen);
            }
        }
    }

    private Protocol() {
        this.types = new Schema.Names();
        this.messages = new LinkedHashMap();
        this.props = new Schema.Props(PROTOCOL_RESERVED);
    }

    public Protocol(String name, String doc, String namespace) {
        this.types = new Schema.Names();
        this.messages = new LinkedHashMap();
        this.props = new Schema.Props(PROTOCOL_RESERVED);
        this.name = name;
        this.doc = doc;
        this.namespace = namespace;
    }

    public Protocol(String name, String namespace) {
        this(name, null, namespace);
    }

    public String getName() {
        return this.name;
    }

    public String getNamespace() {
        return this.namespace;
    }

    public String getDoc() {
        return this.doc;
    }

    public Collection<Schema> getTypes() {
        return this.types.values();
    }

    public Schema getType(String name) {
        return this.types.get((Object) name);
    }

    public void setTypes(Collection<Schema> newTypes) {
        this.types = new Schema.Names();
        for (Schema s : newTypes) {
            this.types.add(s);
        }
    }

    public Map<String, Message> getMessages() {
        return this.messages;
    }

    @Deprecated
    public Message createMessage(String name, String doc, Schema request) {
        return createMessage(name, doc, new LinkedHashMap(), request);
    }

    public Message createMessage(String name, String doc, Map<String, String> propMap, Schema request) {
        return new Message(name, doc, propMap, request);
    }

    @Deprecated
    public Message createMessage(String name, String doc, Schema request, Schema response, Schema errors) {
        return createMessage(name, doc, new LinkedHashMap(), request, response, errors);
    }

    public Message createMessage(String name, String doc, Map<String, String> propMap, Schema request, Schema response, Schema errors) {
        return new TwoWayMessage(name, doc, propMap, request, response, errors);
    }

    public synchronized String getProp(String name) {
        return this.props.get(name);
    }

    public synchronized void addProp(String name, String value) {
        this.props.add(name, value);
    }

    public Map<String, String> getProps() {
        return Collections.unmodifiableMap(this.props);
    }

    public boolean equals(Object o) {
        if (o == this) {
            return true;
        }
        if (o instanceof Protocol) {
            Protocol that = (Protocol) o;
            return this.name.equals(that.name) && this.namespace.equals(that.namespace) && this.types.equals(that.types) && this.messages.equals(that.messages) && this.props.equals(this.props);
        }
        return false;
    }

    public int hashCode() {
        return this.name.hashCode() + this.namespace.hashCode() + this.types.hashCode() + this.messages.hashCode() + this.props.hashCode();
    }

    public String toString() {
        return toString(false);
    }

    public String toString(boolean pretty) {
        try {
            StringWriter writer = new StringWriter();
            JsonGenerator gen = Schema.FACTORY.createJsonGenerator(writer);
            if (pretty) {
                gen.useDefaultPrettyPrinter();
            }
            toJson(gen);
            gen.flush();
            return writer.toString();
        } catch (IOException e) {
            throw new AvroRuntimeException(e);
        }
    }

    void toJson(JsonGenerator gen) throws IOException {
        this.types.space(this.namespace);
        gen.writeStartObject();
        gen.writeStringField("protocol", this.name);
        gen.writeStringField("namespace", this.namespace);
        if (this.doc != null) {
            gen.writeStringField("doc", this.doc);
        }
        this.props.write(gen);
        gen.writeArrayFieldStart("types");
        Schema.Names resolved = new Schema.Names(this.namespace);
        for (Schema type : this.types.values()) {
            if (!resolved.contains(type)) {
                type.toJson(resolved, gen);
            }
        }
        gen.writeEndArray();
        gen.writeObjectFieldStart("messages");
        for (Map.Entry<String, Message> e : this.messages.entrySet()) {
            gen.writeFieldName(e.getKey());
            e.getValue().toJson(gen);
        }
        gen.writeEndObject();
        gen.writeEndObject();
    }

    public byte[] getMD5() {
        if (this.md5 == null) {
            try {
                this.md5 = MessageDigest.getInstance("MD5").digest(toString().getBytes("UTF-8"));
            } catch (Exception e) {
                throw new AvroRuntimeException(e);
            }
        }
        return this.md5;
    }

    public static Protocol parse(File file) throws IOException {
        return parse(Schema.FACTORY.createJsonParser(file));
    }

    public static Protocol parse(InputStream stream) throws IOException {
        return parse(Schema.FACTORY.createJsonParser(stream));
    }

    public static Protocol parse(String string) {
        try {
            return parse(Schema.FACTORY.createJsonParser(new ByteArrayInputStream(string.getBytes("UTF-8"))));
        } catch (IOException e) {
            throw new AvroRuntimeException(e);
        }
    }

    private static Protocol parse(JsonParser parser) {
        try {
            Protocol protocol = new Protocol();
            protocol.parse(Schema.MAPPER.readTree(parser));
            return protocol;
        } catch (IOException e) {
            throw new SchemaParseException(e);
        }
    }

    private void parse(JsonNode json) {
        parseNamespace(json);
        parseName(json);
        parseTypes(json);
        parseMessages(json);
        parseDoc(json);
        parseProps(json);
    }

    private void parseNamespace(JsonNode json) {
        JsonNode nameNode = json.get("namespace");
        if (nameNode != null) {
            this.namespace = nameNode.getTextValue();
            this.types.space(this.namespace);
        }
    }

    private void parseDoc(JsonNode json) {
        this.doc = parseDocNode(json);
    }

    private String parseDocNode(JsonNode json) {
        JsonNode nameNode = json.get("doc");
        if (nameNode == null) {
            return null;
        }
        return nameNode.getTextValue();
    }

    private void parseName(JsonNode json) {
        JsonNode nameNode = json.get("protocol");
        if (nameNode == null) {
            throw new SchemaParseException("No protocol name specified: " + json);
        }
        this.name = nameNode.getTextValue();
    }

    private void parseTypes(JsonNode json) {
        JsonNode defs = json.get("types");
        if (defs != null) {
            if (!defs.isArray()) {
                throw new SchemaParseException("Types not an array: " + defs);
            }
            Iterator i$ = defs.iterator();
            while (i$.hasNext()) {
                JsonNode type = i$.next();
                if (!type.isObject()) {
                    throw new SchemaParseException("Type not an object: " + type);
                }
                Schema.parse(type, this.types);
            }
        }
    }

    private void parseProps(JsonNode json) {
        Iterator<String> i = json.getFieldNames();
        while (i.hasNext()) {
            String p = i.next();
            if (!PROTOCOL_RESERVED.contains(p)) {
                JsonNode prop = json.get(p);
                if (prop.isValueNode() && prop.isTextual()) {
                    addProp(p, prop.getTextValue());
                }
            }
        }
    }

    private void parseMessages(JsonNode json) {
        JsonNode defs = json.get("messages");
        if (defs != null) {
            Iterator<String> i = defs.getFieldNames();
            while (i.hasNext()) {
                String prop = i.next();
                this.messages.put(prop, parseMessage(prop, defs.get(prop)));
            }
        }
    }

    private Message parseMessage(String messageName, JsonNode json) {
        String doc = parseDocNode(json);
        Map<String, String> mProps = new LinkedHashMap<>();
        Iterator<String> i = json.getFieldNames();
        while (i.hasNext()) {
            String p = i.next();
            if (!MESSAGE_RESERVED.contains(p)) {
                JsonNode prop = json.get(p);
                if (prop.isValueNode() && prop.isTextual()) {
                    mProps.put(p, prop.getTextValue());
                }
            }
        }
        JsonNode requestNode = json.get("request");
        if (requestNode == null || !requestNode.isArray()) {
            throw new SchemaParseException("No request specified: " + json);
        }
        List<Schema.Field> fields = new ArrayList<>();
        Iterator i$ = requestNode.iterator();
        while (i$.hasNext()) {
            JsonNode field = i$.next();
            JsonNode fieldNameNode = field.get(TapjoyConstants.TJC_EVENT_IAP_NAME);
            if (fieldNameNode == null) {
                throw new SchemaParseException("No param name: " + field);
            }
            JsonNode fieldTypeNode = field.get(EventDataManager.Events.COLUMN_NAME_TYPE);
            if (fieldTypeNode == null) {
                throw new SchemaParseException("No param type: " + field);
            }
            fields.add(new Schema.Field(fieldNameNode.getTextValue(), Schema.parse(fieldTypeNode, this.types), null, field.get("default")));
        }
        Schema request = Schema.createRecord(fields);
        boolean oneWay = false;
        JsonNode oneWayNode = json.get("one-way");
        if (oneWayNode != null) {
            if (!oneWayNode.isBoolean()) {
                throw new SchemaParseException("one-way must be boolean: " + json);
            }
            oneWay = oneWayNode.getBooleanValue();
        }
        JsonNode responseNode = json.get("response");
        if (!oneWay && responseNode == null) {
            throw new SchemaParseException("No response specified: " + json);
        }
        JsonNode decls = json.get("errors");
        if (oneWay) {
            if (decls != null) {
                throw new SchemaParseException("one-way can't have errors: " + json);
            }
            if (responseNode != null && Schema.parse(responseNode, this.types).getType() != Schema.Type.NULL) {
                throw new SchemaParseException("One way response must be null: " + json);
            }
            return new Message(messageName, doc, mProps, request);
        }
        Schema response = Schema.parse(responseNode, this.types);
        List<Schema> errs = new ArrayList<>();
        errs.add(SYSTEM_ERROR);
        if (decls != null) {
            if (!decls.isArray()) {
                throw new SchemaParseException("Errors not an array: " + json);
            }
            Iterator i$2 = decls.iterator();
            while (i$2.hasNext()) {
                JsonNode decl = i$2.next();
                String name = decl.getTextValue();
                Schema schema = this.types.get((Object) name);
                if (schema == null) {
                    throw new SchemaParseException("Undefined error: " + name);
                }
                if (!schema.isError()) {
                    throw new SchemaParseException("Not an error: " + name);
                }
                errs.add(schema);
            }
        }
        return new TwoWayMessage(messageName, doc, mProps, request, response, Schema.createUnion(errs));
    }

    public static void main(String[] args) throws Exception {
        System.out.println(parse(new File(args[0])));
    }
}
