package com.arellomobile.android.push;

import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import android.os.AsyncTask;
import android.os.Bundle;
import android.os.Handler;
import com.arellomobile.android.push.exception.PushWooshException;
import com.arellomobile.android.push.preference.SoundType;
import com.arellomobile.android.push.preference.VibrateType;
import com.arellomobile.android.push.tags.SendPushTagsAsyncTask;
import com.arellomobile.android.push.tags.SendPushTagsCallBack;
import com.arellomobile.android.push.utils.GeneralUtils;
import com.arellomobile.android.push.utils.PreferenceUtils;
import com.arellomobile.android.push.utils.WorkerTask;
import com.arellomobile.android.push.utils.executor.ExecutorHelper;
import com.google.android.gcm.GCMRegistrar;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

/* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
public class PushManager {
    private static final String HTML_URL_FORMAT = "https://cp.pushwoosh.com/content/%s";
    public static final String PUSH_RECEIVE_EVENT = "PUSH_RECEIVE_EVENT";
    public static final String REGISTER_ERROR_EVENT = "REGISTER_ERROR_EVENT";
    public static final String REGISTER_EVENT = "REGISTER_EVENT";
    public static final String UNREGISTER_ERROR_EVENT = "UNREGISTER_ERROR_EVENT";
    public static final String UNREGISTER_EVENT = "UNREGISTER_EVENT";
    private static AsyncTask<Void, Void, Void> mRegistrationAsyncTask;
    static volatile String mSenderId;
    static Boolean sSimpleNotification;
    private volatile String mAppId;
    private Context mContext;
    private Bundle mLastBundle;
    public static int MESSAGE_ID = 1001;
    static SoundType sSoundType = SoundType.DEFAULT_MODE;
    static VibrateType sVibrateType = VibrateType.DEFAULT_MODE;
    private static final Object mSyncObj = new Object();

    Context getContext() {
        return this.mContext;
    }

    /* JADX INFO: Access modifiers changed from: package-private */
    public PushManager(Context context) {
        GeneralUtils.checkNotNull(context, "context");
        this.mContext = context;
        this.mAppId = PreferenceUtils.getApplicationId(context);
        mSenderId = PreferenceUtils.getSenderId(context);
    }

    public PushManager(Context context, String appId, String senderId) {
        this(context);
        this.mAppId = appId;
        mSenderId = senderId;
        PreferenceUtils.setApplicationId(context, this.mAppId);
        PreferenceUtils.setSenderId(context, senderId);
    }

    public void onStartup(Context context) {
        GeneralUtils.checkNotNullOrEmpty(this.mAppId, "mAppId");
        GeneralUtils.checkNotNullOrEmpty(mSenderId, "mSenderId");
        GCMRegistrar.checkDevice(context);
        GCMRegistrar.checkManifest(context);
        String regId = GCMRegistrar.getRegistrationId(context);
        if (!regId.equals("")) {
            if (!(context instanceof Activity) || !((Activity) context).getIntent().hasExtra(PUSH_RECEIVE_EVENT)) {
                String oldAppId = PreferenceUtils.getApplicationId(context);
                if (!oldAppId.equals(this.mAppId)) {
                    registerOnPushWoosh(context, regId);
                    return;
                } else if (neededToRequestPushWooshServer(context)) {
                    registerOnPushWoosh(context, regId);
                    return;
                } else {
                    PushEventsTransmitter.onRegistered(context, regId);
                    return;
                }
            }
            return;
        }
        GCMRegistrar.register(context, new String[]{mSenderId});
    }

    public void startTrackingGeoPushes() {
        this.mContext.startService(new Intent(this.mContext, GeoLocationService.class));
    }

    public void stopTrackingGeoPushes() {
        this.mContext.stopService(new Intent(this.mContext, GeoLocationService.class));
    }

    public void unregister() {
        cancelPrevRegisterTask();
        GCMRegistrar.unregister(this.mContext);
    }

    public String getCustomData() {
        if (this.mLastBundle == null) {
            return null;
        }
        return this.mLastBundle.getString("u");
    }

    public static Map<String, String> sendTagsFromBG(Context context, Map<String, Object> tags) throws PushWooshException {
        Map<String, String> wrongTags = new HashMap<>();
        try {
            JSONArray wrongTagsArray = DeviceFeature2_5.sendTags(context, tags);
            for (int i = 0; i < wrongTagsArray.length(); i++) {
                JSONObject reason = wrongTagsArray.getJSONObject(i);
                wrongTags.put(reason.getString("tag"), reason.getString("reason"));
            }
            return wrongTags;
        } catch (Exception e) {
            throw new PushWooshException(e);
        }
    }

    public static void sendTagsFromUI(Context context, Map<String, Object> tags, SendPushTagsCallBack callBack) {
        new SendPushTagsAsyncTask(context, callBack).execute(new Map[]{tags});
    }

    public static void sendTags(final Context context, final Map<String, Object> tags, final SendPushTagsCallBack callBack) {
        Handler handler = new Handler(context.getMainLooper());
        handler.post(new Runnable() { // from class: com.arellomobile.android.push.PushManager.1
            @Override // java.lang.Runnable
            public void run() {
                new SendPushTagsAsyncTask(context, callBack).execute(new Map[]{tags});
            }
        });
    }

    public void setMultiNotificationMode() {
        sSimpleNotification = false;
    }

    public void setSimpleNotificationMode() {
        sSimpleNotification = true;
    }

    public void setSoundNotificationType(SoundType soundNotificationType) {
        sSoundType = soundNotificationType;
    }

    public void setVibrateNotificationType(VibrateType vibrateNotificationType) {
        sVibrateType = vibrateNotificationType;
    }

    /* JADX INFO: Access modifiers changed from: package-private */
    public boolean onHandlePush(Activity activity) {
        Bundle pushBundle = activity.getIntent().getBundleExtra("pushBundle");
        if (pushBundle == null || this.mContext == null) {
            return false;
        }
        this.mLastBundle = pushBundle;
        JSONObject dataObject = new JSONObject();
        try {
            if (pushBundle.containsKey("title")) {
                dataObject.put("title", pushBundle.get("title"));
            }
            if (pushBundle.containsKey("u")) {
                dataObject.put("userdata", new JSONObject(pushBundle.getString("u")));
            }
        } catch (JSONException e) {
        }
        PushEventsTransmitter.onMessageReceive(this.mContext, dataObject.toString());
        String url = (String) pushBundle.get("h");
        if (url != null) {
            String url2 = String.format(HTML_URL_FORMAT, url);
            Intent intent = new Intent(activity, PushWebview.class);
            intent.putExtra("url", url2);
            activity.startActivity(intent);
        }
        sendPushStat(this.mContext, pushBundle.getString("p"));
        return true;
    }

    private boolean neededToRequestPushWooshServer(Context context) {
        Calendar nowTime = Calendar.getInstance();
        Calendar dayBefore = Calendar.getInstance();
        dayBefore.roll(5, false);
        Calendar lastPushWooshRegistrationTime = Calendar.getInstance();
        lastPushWooshRegistrationTime.setTime(new Date(PreferenceUtils.getLastRegistration(context)));
        return (dayBefore.before(lastPushWooshRegistrationTime) && lastPushWooshRegistrationTime.before(nowTime)) ? false : true;
    }

    private void registerOnPushWoosh(Context context, String regId) {
        cancelPrevRegisterTask();
        mRegistrationAsyncTask = getRegisterAsyncTask(context, regId);
        ExecutorHelper.executeAsyncTask(mRegistrationAsyncTask);
    }

    private void sendPushStat(Context context, final String hash) {
        try {
            AsyncTask<Void, Void, Void> task = new WorkerTask(context) { // from class: com.arellomobile.android.push.PushManager.2
                @Override // com.arellomobile.android.push.utils.WorkerTask
                protected void doWork(Context context2) {
                    DeviceFeature2_5.sendPushStat(context2, hash);
                }
            };
            ExecutorHelper.executeAsyncTask(task);
        } catch (Throwable th) {
            DeviceFeature2_5.sendPushStat(context, hash);
        }
    }

    private AsyncTask<Void, Void, Void> getRegisterAsyncTask(Context context, final String regId) {
        try {
            return new WorkerTask(context) { // from class: com.arellomobile.android.push.PushManager.3
                @Override // com.arellomobile.android.push.utils.WorkerTask
                protected void doWork(Context context2) {
                    DeviceRegistrar.registerWithServer(PushManager.this.mContext, regId);
                }
            };
        } catch (Throwable th) {
            DeviceRegistrar.registerWithServer(context, regId);
            return null;
        }
    }

    private void cancelPrevRegisterTask() {
        synchronized (mSyncObj) {
            if (mRegistrationAsyncTask != null) {
                mRegistrationAsyncTask.cancel(true);
            }
            mRegistrationAsyncTask = null;
        }
    }
}
