package com.arellomobile.android.push;

import android.app.Service;
import android.content.Context;
import android.content.Intent;
import android.location.Location;
import android.location.LocationListener;
import android.location.LocationManager;
import android.os.AsyncTask;
import android.os.Bundle;
import android.os.IBinder;
import android.os.PowerManager;
import android.util.Log;
import com.arellomobile.android.push.data.PushZoneLocation;
import com.arellomobile.android.push.utils.WorkerTask;
import com.arellomobile.android.push.utils.executor.ExecutorHelper;
import com.google.android.gcm.GCMRegistrar;

/* loaded from: /storage/emulated/0/Documents/jadec/sources/com.UBI.A90.WW/dex-files/0.dex */
public class GeoLocationService extends Service {
    private static final int TEN_METERS = 10;
    private static final int TEN_SECONDS = 10000;
    private static final int TWO_MINUTES = 120000;
    private LocationManager mLocationManager;
    private long mMinDistance;
    private Location mOldLocation;
    private PowerManager.WakeLock mWakeLock;
    private final Object mSyncObject = new Object();
    private boolean mIfUpdating = false;
    private final LocationListener mListener = new LocationListener() { // from class: com.arellomobile.android.push.GeoLocationService.1
        @Override // android.location.LocationListener
        public void onLocationChanged(Location location) {
            GeoLocationService.this.updateLocation(location);
        }

        @Override // android.location.LocationListener
        public void onProviderDisabled(String provider) {
        }

        @Override // android.location.LocationListener
        public void onProviderEnabled(String provider) {
        }

        @Override // android.location.LocationListener
        public void onStatusChanged(String provider, int status, Bundle extras) {
        }
    };

    @Override // android.app.Service
    public int onStartCommand(Intent intent, int flags, int startId) {
        super.onStartCommand(intent, flags, startId);
        initService();
        return 1;
    }

    @Override // android.app.Service
    public void onDestroy() {
        super.onDestroy();
        this.mLocationManager.removeUpdates(this.mListener);
        this.mWakeLock.release();
        this.mWakeLock = null;
        this.mLocationManager = null;
    }

    @Override // android.app.Service
    public IBinder onBind(Intent intent) {
        return null;
    }

    private void initService() {
        Location gpsLocation;
        Location networkLocation;
        PowerManager pm = (PowerManager) getSystemService("power");
        this.mWakeLock = pm.newWakeLock(1, getClass().getName());
        this.mWakeLock.acquire();
        this.mLocationManager = (LocationManager) getSystemService("location");
        synchronized (this.mSyncObject) {
            this.mLocationManager.removeUpdates(this.mListener);
            gpsLocation = requestUpdatesFromProvider("gps", 10L);
            networkLocation = requestUpdatesFromProvider("network", 10L);
        }
        if (gpsLocation != null && networkLocation != null) {
            updateLocation(getBetterLocation(gpsLocation, networkLocation));
        } else if (gpsLocation != null) {
            updateLocation(gpsLocation);
        } else if (networkLocation != null) {
            updateLocation(networkLocation);
        }
    }

    /* JADX INFO: Access modifiers changed from: private */
    public void updateLocation(final Location location) {
        synchronized (this.mSyncObject) {
            if (!this.mIfUpdating && GCMRegistrar.isRegisteredOnServer(this)) {
                if (this.mOldLocation == null || location.distanceTo(this.mOldLocation) >= ((float) this.mMinDistance)) {
                    this.mOldLocation = location;
                    this.mLocationManager.removeUpdates(this.mListener);
                    this.mIfUpdating = true;
                    AsyncTask<Void, Void, Void> task = new WorkerTask(this) { // from class: com.arellomobile.android.push.GeoLocationService.2
                        protected PushZoneLocation mZoneLocation;

                        @Override // com.arellomobile.android.push.utils.WorkerTask
                        protected void doWork(Context context) throws Exception {
                            this.mZoneLocation = DeviceFeature2_5.getNearestZone(context, location);
                        }

                        /* JADX INFO: Access modifiers changed from: protected */
                        @Override // android.os.AsyncTask
                        public void onPostExecute(Void result) {
                            super.onPostExecute((AnonymousClass2) result);
                            synchronized (GeoLocationService.this.mSyncObject) {
                                long distance = 0;
                                if (this.mZoneLocation != null) {
                                    distance = this.mZoneLocation.getDistanceTo();
                                }
                                long distance2 = Math.max(10L, distance);
                                GeoLocationService.this.requestUpdatesFromProvider("gps", distance2);
                                GeoLocationService.this.requestUpdatesFromProvider("network", distance2);
                                GeoLocationService.this.mIfUpdating = false;
                            }
                        }
                    };
                    ExecutorHelper.executeAsyncTask(task);
                }
            }
        }
    }

    /* JADX INFO: Access modifiers changed from: private */
    public Location requestUpdatesFromProvider(String provider, long distance) {
        this.mMinDistance = distance;
        try {
            if (!this.mLocationManager.isProviderEnabled(provider)) {
                return null;
            }
            this.mLocationManager.requestLocationUpdates(provider, 10000L, (float) distance, this.mListener);
            Location location = this.mLocationManager.getLastKnownLocation(provider);
            return location;
        } catch (Exception e) {
            Log.e(getClass().getName(), "Check ACCESS_FINE_LOCATION permission", e);
            return null;
        }
    }

    protected Location getBetterLocation(Location newLocation, Location currentBestLocation) {
        if (currentBestLocation != null) {
            long timeDelta = newLocation.getTime() - currentBestLocation.getTime();
            boolean isSignificantlyNewer = timeDelta > 120000;
            boolean isSignificantlyOlder = timeDelta < -120000;
            boolean isNewer = timeDelta > 0;
            if (!isSignificantlyNewer) {
                if (isSignificantlyOlder) {
                    return currentBestLocation;
                }
                int accuracyDelta = (int) (newLocation.getAccuracy() - currentBestLocation.getAccuracy());
                boolean isLessAccurate = accuracyDelta > 0;
                boolean isMoreAccurate = accuracyDelta < 0;
                boolean isSignificantlyLessAccurate = accuracyDelta > 200;
                boolean isFromSameProvider = isSameProvider(newLocation.getProvider(), currentBestLocation.getProvider());
                if (!isMoreAccurate) {
                    if (!isNewer || isLessAccurate) {
                        return (isNewer && !isSignificantlyLessAccurate && isFromSameProvider) ? newLocation : currentBestLocation;
                    }
                    return newLocation;
                }
                return newLocation;
            }
            return newLocation;
        }
        return newLocation;
    }

    private boolean isSameProvider(String provider1, String provider2) {
        if (provider1 == null) {
            return provider2 == null;
        }
        return provider1.equals(provider2);
    }
}
