#!/bin/bash
#

# shellcheck disable=SC2154,SC2086,SC2059

###################
# flash RW_LEGACY #
###################
function flash_rwlegacy()
{
	#set working dir
	cd /tmp || { exit_red "Error changing to tmp dir; cannot proceed"; return 1; }

	# set dev mode legacy boot / AltFw flags
	if [ "${isChromeOS}" = true ]; then
		crossystem dev_boot_legacy=1 > /dev/null 2>&1
		crossystem dev_boot_altfw=1 > /dev/null 2>&1
	fi

	#determine proper file
	if [ "$device" = "link" ]; then
		rwlegacy_file=$seabios_link
	elif [[ "$isHswBox" = true || "$isBdwBox" = true ]]; then
		rwlegacy_file=$seabios_hswbdw_box
	elif [[ "$isHswBook" = true || "$isBdwBook" = true ]]; then
		rwlegacy_file=$seabios_hswbdw_book
	elif [ "$isByt" = true ]; then
		rwlegacy_file=$seabios_baytrail
	elif [ "$isBsw" = true ]; then
		rwlegacy_file=$seabios_braswell
	elif [ "$isSkl" = true ]; then
		rwlegacy_file=$seabios_skylake
	elif [ "$isApl" = true ]; then
		rwlegacy_file=$seabios_apl
	elif [ "$kbl_use_rwl18" = true ]; then
		rwlegacy_file=$seabios_kbl_18
	elif [ "$isKbl" = true ]; then
		rwlegacy_file=$seabios_kbl
	elif [ "$isWhl" = true ]; then
		rwlegacy_file=$rwl_altfw_whl
	elif [ "$device" = "drallion" ]; then
		rwlegacy_file=$rwl_altfw_drallion
	elif [ "$isCmlBox" = true ]; then
		rwlegacy_file=$rwl_altfw_cml
	elif [ "$isJsl" = true ]; then
		rwlegacy_file=$rwl_altfw_jsl
	elif [ "$isTgl" = true ]; then
		rwlegacy_file=$rwl_altfw_tgl
	elif [ "$isGlk" = true ]; then
		rwlegacy_file=$rwl_altfw_glk
	elif [ "$isAdl" = true ]; then
		rwlegacy_file=$rwl_altfw_adl
	elif [ "$isAdlN" = true ]; then
		rwlegacy_file=$rwl_altfw_adl_n
	elif [ "$isStr" = true ]; then
		rwlegacy_file=$rwl_altfw_stoney
	elif [ "$isPco" = true ]; then
		rwlegacy_file=$rwl_altfw_pco
	elif [ "$isCzn" = true ]; then
		rwlegacy_file=$rwl_altfw_czn
	elif [ "$isMdn" = true ]; then
		rwlegacy_file=$rwl_altfw_mdn
	else
		echo_red "Unknown or unsupported device (${device}); cannot update RW_LEGACY firmware."
		echo_red "If your device is listed as supported on https://mrchromebox.tech/#devices,\n
then email MrChromebox@gmail.com  and include a screenshot of the main menu."
		read -rep "Press enter to return to the main menu"
		return 1
	fi

	if [[ "$rwlegacy_file" = *"altfw"* ]]; then
		echo_green "\nInstall/Update RW_LEGACY Firmware (AltFw / edk2)"
	else
		echo_green "\nInstall/Update RW_LEGACY Firmware (Legacy BIOS / SeaBIOS)"
	fi

	echo_yellow "
NOTE: RW_LEGACY firmware cannot be used to run Windows. Period.
If you are looking to run Windows, see the documentation on coolstar.org.
MrChromebox does not provide any support for running Windows."
		REPLY=""
		read -rep "Press Y to continue or any other key to return to the main menu. "
		[[ "$REPLY" = "y" || "$REPLY" = "Y" ]] || return

	preferUSB=false
	if [[ "$rwlegacy_file" != *"altfw"* ]]; then
		echo -e ""
		#USB boot priority
		echo_yellow "Default to booting from USB?"
		read -rep "If N, always boot from internal storage unless selected from boot menu. [y/N] "
		[[ "$REPLY" = "y" || "$REPLY" = "Y" ]] && preferUSB=true
		echo -e ""
	fi

	if ! [ -f "${rwlegacy_source}${rwlegacy_file}" ]; then
		exit_red "RW_LEGACY payload not found :("
		return 1
	fi

	#preferUSB?
	if [ "$preferUSB" = true  ]; then
		if ! $CURL -sLo bootorder "${cbfs_source}bootorder.usb"; then
			echo_red "Unable to download bootorder file; boot order cannot be changed."
		else
			${cbfstoolcmd} "${rwlegacy_source}${rwlegacy_file}" remove -n bootorder > /dev/null 2>&1
			${cbfstoolcmd} "${rwlegacy_source}${rwlegacy_file}" add -n bootorder -f /tmp/bootorder -t raw > /dev/null 2>&1
		fi
	fi

	#flash updated RW_LEGACY firmware
	echo_yellow "Installing RW_LEGACY firmware"
	if ! ${flashromcmd} -w -i RW_LEGACY:"${rwlegacy_source}${rwlegacy_file}" -o /tmp/flashrom.log > /dev/null 2>&1; then
		cat /tmp/flashrom.log
		echo_red "An error occurred flashing the RW_LEGACY firmware."
	else
		echo_green "RW_LEGACY firmware successfully installed/updated."
		# update firmware type
		firmwareType="Stock ChromeOS w/RW_LEGACY"
		#Prevent from trying to boot stock ChromeOS install
		[[ "$boot_mounted" = true ]] && rm -rf /tmp/boot/syslinux > /dev/null 2>&1
	fi

	read -rep "Press [Enter] to return to the main menu."
}


#############################
# Install Full ROM Firmware #
#############################
function flash_full_rom()
{
	echo "Removed..."
	read -rep "Press [Enter] to return to the main menu."
}

#########################
# Downgrade Touchpad FW #
#########################
function downgrade_touchpad_fw()
{
	echo "Removed..."
	read -rep "Press [Enter] to return to the main menu."
}

##########################
# Restore Stock Firmware #
##########################
function restore_stock_firmware()
{
	echo "Removed..."
	read -rep "Press [Enter] to return to the main menu."
}

function restore_fw_from_usb()
{
	echo "Removed..."
	read -rep "Press [Enter] to return to the main menu."
}

function restore_fw_from_shellball()
{
	echo "Removed..."
	read -rep "Press [Enter] to return to the main menu."
}

function restore_fw_from_recovery()
{
	echo "Removed..."
	read -rep "Press [Enter] to return to the main menu."
}

####################
# Set Boot Options #
####################
function set_boot_options()
{
	# set boot options via firmware boot flags

	# ensure hardware write protect disabled
	[[ "$wpEnabled" = true ]] && { exit_red  "\nHardware write-protect enabled, cannot set Boot Options / GBB Flags."; return 1; }

	echo_green "\nSet Firmware Boot Options (GBB Flags)"
	echo_yellow "Select your preferred boot delay and default boot option.
You can always override the default using [CTRL+D] or
[CTRL+L] on the Developer Mode boot screen"

	echo -e "1) Short boot delay (1s) + Legacy Boot/AltFw default
2) Long boot delay (30s) + Legacy Boot/AltFw default
3) Short boot delay (1s) + ChromeOS default
4) Long boot delay (30s) + ChromeOS default
5) Reset to factory default
6) Cancel/exit
"
	local _flags=0x0
	while :
	do
		read -rep "? " n
		case $n in
			1) _flags=0x4A9; break;;
			2) _flags=0x4A8; break;;
			3) _flags=0xA9; break;;
			4) _flags=0xA8; break;;
			5) _flags=0x0; break;;
			6) read -rep "Press [Enter] to return to the main menu."; break;;
			*) echo -e "invalid option";;
		esac
	done
	[[ $n -eq 6 ]] && return
	echo_yellow "\nSetting boot options..."

	#disable software write-protect
	if ! ${flashromcmd} --wp-disable > /dev/null 2>&1; then
		exit_red "Error disabling software write-protect; unable to set GBB flags."; return 1
	fi
	if ! ${flashromcmd} -r -i GBB:/tmp/gbb.temp > /dev/null 2>&1; then
		exit_red "\nError reading firmware (non-stock?); unable to set boot options."; return 1
	fi
	if ! ${gbbutilitycmd} --set --flags="${_flags}" /tmp/gbb.temp > /dev/null; then
		exit_red "\nError setting boot options."; return 1
	fi
	if ! ${flashromcmd} -w -i GBB:/tmp/gbb.temp > /dev/null 2>&1; then
		exit_red "\nError writing back firmware; unable to set boot options."; return 1
	fi

	echo_green "\nFirmware Boot options successfully set."

	read -rep "Press [Enter] to return to the main menu."
}


###################
# Set Hardware ID #
###################
function set_hwid()
{
	# set HWID using gbb_utility

	# ensure hardware write protect disabled
	[[ "$wpEnabled" = true ]] && { exit_red  "\nHardware write-protect enabled, cannot set HWID."; return 1; }

	echo_green "Set Hardware ID (HWID) using gbb_utility"

	#get current HWID
	_hwid="$(crossystem hwid)" >/dev/null 2>&1
	if [[ "$_hwid" != "" ]]; then
		echo_yellow "Current HWID is $_hwid"
	fi

	echo_yellow "Are you sure you know what you're doing here?
Changing this is not normally needed, and if you mess it up,
MrChromebox is not going to help you fix it. This won't let
you run a different/newer version of ChromeOS.
Proceed at your own risk."

	read -rep "Really change your HWID? [y/N] " confirm
	[[ "$confirm" = "Y" || "$confirm" = "y" ]] || return

	read -rep "This is serious. Are you really sure? [y/N] " confirm
	[[ "$confirm" = "Y" || "$confirm" = "y" ]] || return

	read -rep "Enter a new HWID (use all caps): " hwid
	echo -e ""
	read -rep "Confirm changing HWID to $hwid [y/N] " confirm
	if [[ "$confirm" = "Y" || "$confirm" = "y" ]]; then
		echo_yellow "\nSetting hardware ID..."
		#disable software write-protect
		if ! ${flashromcmd} --wp-disable > /dev/null 2>&1; then
			exit_red "Error disabling software write-protect; unable to set HWID."; return 1
		fi
		if ! ${flashromcmd} -r -i GBB:/tmp/gbb.temp > /dev/null 2>&1; then
			exit_red "\nError reading firmware (non-stock?); unable to set HWID."; return 1
		fi
		if ! ${gbbutilitycmd} --set --hwid="${hwid}" /tmp/gbb.temp > /dev/null 2>&1; then
			exit_red "\nError setting HWID."; return 1
		fi
		if ! ${flashromcmd} -w -i GBB:/tmp/gbb.temp > /dev/null 2>&1; then
			exit_red "\nError writing back firmware; unable to set HWID."; return 1
		fi

		echo_green "Hardware ID successfully set."
	fi
	read -rep "Press [Enter] to return to the main menu."
}


##################
# Remove Bitmaps #
##################
function remove_bitmaps()
{
	# remove bitmaps from GBB using gbb_utility

	# ensure hardware write protect disabled
	[[ "$wpEnabled" = true ]] && { exit_red  "\nHardware write-protect enabled, cannot remove bitmaps."; return 1; }

	echo_green "\nRemove ChromeOS Boot Screen Bitmaps"

	read -rep "Confirm removing ChromeOS bitmaps? [y/N] " confirm
	if [[ "$confirm" = "Y" || "$confirm" = "y" ]]; then
		echo_yellow "\nRemoving bitmaps..."
		#disable software write-protect
		if ! ${flashromcmd} --wp-disable > /dev/null 2>&1; then
			exit_red "Error disabling software write-protect; unable to remove bitmaps."; return 1
		fi
		#read GBB region
		if ! ${flashromcmd} -r -i GBB:/tmp/gbb.temp > /dev/null 2>&1; then
			exit_red "\nError reading firmware (non-stock?); unable to remove bitmaps."; return 1
		fi
		touch /tmp/null-images > /dev/null 2>&1
		#set bitmaps to null
		if ! ${gbbutilitycmd} --set --bmpfv=/tmp/null-images /tmp/gbb.temp > /dev/null 2>&1; then
			exit_red "\nError removing bitmaps."; return 1
		fi
		#flash back to board
		if ! ${flashromcmd} -w -i GBB:/tmp/gbb.temp > /dev/null 2>&1; then
			exit_red "\nError writing back firmware; unable to remove bitmaps."; return 1
		fi

		echo_green "ChromeOS bitmaps successfully removed."
	fi
	read -rep "Press [Enter] to return to the main menu."
}


##################
# Restore Bitmaps #
##################
function restore_bitmaps()
{
	# restore bitmaps from GBB using gbb_utility

	# ensure hardware write protect disabled
	[[ "$wpEnabled" = true ]] && { exit_red  "\nHardware write-protect enabled, cannot restore bitmaps."; return 1; }

	echo_green "\nRestore ChromeOS Boot Screen Bitmaps"

	read -rep "Confirm restoring ChromeOS bitmaps? [y/N] " confirm
	if [[ "$confirm" = "Y" || "$confirm" = "y" ]]; then
		echo_yellow "\nRestoring bitmaps..."
		#disable software write-protect
		if ! ${flashromcmd} --wp-disable > /dev/null 2>&1; then
			exit_red "Error disabling software write-protect; unable to restore bitmaps."; return 1
		fi
		#download shellball
		if ! $CURL -sLo /tmp/shellball.rom ${shellball_source}shellball.${device}.bin; then
			exit_red "Error downloading shellball; unable to restore bitmaps."; return 1
		fi
		#extract GBB region, bitmaps
		if ! ${cbfstoolcmd} /tmp/shellball.rom read -r GBB -f gbb.new >/dev/null 2>&1; then
			exit_red "Error extracting GBB region from shellball; unable to restore bitmaps."; return 1
		fi
		if ! ${flashromcmd} -r -i GBB:/tmp/gbb.temp > /dev/null 2>&1; then
			exit_red "\nError reading firmware (non-stock?); unable to restore bitmaps."; return 1
		fi
		#inject bitmaps into GBB
		if ! ${gbbutilitycmd} --get --bmpfv=/tmp/bmpfv /tmp/gbb.new > /dev/null && \
				${gbbutilitycmd} --set --bmpfv=/tmp/bmpfv /tmp/gbb.temp > /dev/null; then
			exit_red "\nError restoring bitmaps."; return 1
		fi
		#flash back to device
		if ! ${flashromcmd} -w -i GBB:/tmp/gbb.temp > /dev/null 2>&1; then
			exit_red "\nError writing back firmware; unable to restore bitmaps."; return 1
		fi

		echo_green "ChromeOS bitmaps successfully restored."
	fi
	read -rep "Press [Enter] to return to the main menu."
}

###############
# Clear NVRAM #
###############
function clear_nvram()
{
	echo "Removed..."
	read -rep "Press [Enter] to return to the main menu."
}

########################
# Firmware Update Menu #
########################
function menu_fwupdate() {

	stock_menu
}

function show_header() {
	printf "\ec"
	echo -e "${NORMAL}\n ChromeOS Device Firmware Utility Script ${script_date} ${NORMAL}"
	echo -e "${NORMAL} (c) Mr Chromebox <mrchromebox@gmail.com> (modded by OlyB for sh1mmer) ${NORMAL}"
	echo -e "${MENU}*********************************************************${NORMAL}"
	echo -e "${MENU}**${NUMBER}   Device: ${NORMAL}${deviceDesc} (${boardName^^})"
	echo -e "${MENU}**${NUMBER} Platform: ${NORMAL}$deviceCpuType"
	echo -e "${MENU}**${NUMBER}  Fw Type: ${NORMAL}$firmwareType"
	echo -e "${MENU}**${NUMBER}   Fw Ver: ${NORMAL}$fwVer ($fwDate)"
	if [[ $isUEFI = true && $hasUEFIoption = true ]]; then
		# check if update available
		curr_yy=$(echo $fwDate | cut -f 3 -d '/')
		curr_mm=$(echo $fwDate | cut -f 1 -d '/')
		curr_dd=$(echo $fwDate | cut -f 2 -d '/')
		eval coreboot_file=$`echo "coreboot_uefi_${device}"`
		date=$(echo $coreboot_file | grep -o "mrchromebox.*" | cut -f 2 -d '_' | cut -f 1 -d '.')
		uefi_yy=$(echo $date | cut -c1-4)
		uefi_mm=$(echo $date | cut -c5-6)
		uefi_dd=$(echo $date | cut -c7-8)
		if [[ ("$firmwareType" != *"pending"*) && (($uefi_yy > $curr_yy) || \
			("$uefi_yy" = "$curr_yy" && "$uefi_mm" > "$curr_mm") || \
			("$uefi_yy" = "$curr_yy" && "$uefi_mm" = "$curr_mm" && "$uefi_dd" > "$curr_dd")) ]]; then
			echo -e "${MENU}**${NORMAL}           ${GREEN_TEXT}Update Available ($uefi_mm/$uefi_dd/$uefi_yy)${NORMAL}"
		fi
	fi
	if [ "$wpEnabled" = true ]; then
		echo -e "${MENU}**${NUMBER}    Fw WP: ${RED_TEXT}Enabled${NORMAL}"
		WP_TEXT=${RED_TEXT}
	else
		echo -e "${MENU}**${NUMBER}    Fw WP: ${NORMAL}Disabled"
		WP_TEXT=${GREEN_TEXT}
	fi
	echo -e "${MENU}*********************************************************${NORMAL}"
}

function stock_menu() {

	show_header

	if [[ "$unlockMenu" = true || ( "$isFullRom" = false && "$isBootStub" = false && "$isUnsupported" = false \
			&& "$isCmlBook" = false ) ]]; then
		echo -e "${MENU}**${WP_TEXT}     ${NUMBER} 1)${MENU} Install/Update RW_LEGACY Firmware ${NORMAL}"
	else
		echo -e "${GRAY_TEXT}**     ${GRAY_TEXT} 1)${GRAY_TEXT} Install/Update RW_LEGACY Firmware ${NORMAL}"
	fi

	if [[ "$unlockMenu" = true || ( "$isFullRom" = false && "$isBootStub" = false ) ]]; then
		echo -e "${MENU}**${WP_TEXT} [WP]${NUMBER} 3)${MENU} Set Boot Options (GBB flags) ${NORMAL}"
		echo -e "${MENU}**${WP_TEXT} [WP]${NUMBER} 4)${MENU} Set Hardware ID (HWID) ${NORMAL}"
	else
		echo -e "${GRAY_TEXT}**     ${GRAY_TEXT} 3)${GRAY_TEXT} Set Boot Options (GBB flags)${NORMAL}"
		echo -e "${GRAY_TEXT}**     ${GRAY_TEXT} 4)${GRAY_TEXT} Set Hardware ID (HWID) ${NORMAL}"
	fi
	if [[ "$unlockMenu" = true || ( "$isFullRom" = false && "$isBootStub" = false && \
		("$isHsw" = true || "$isBdw" = true || "$isByt" = true || "$isBsw" = true )) ]]; then
		echo -e "${MENU}**${WP_TEXT} [WP]${NUMBER} 5)${MENU} Remove ChromeOS Bitmaps ${NORMAL}"
		echo -e "${MENU}**${WP_TEXT} [WP]${NUMBER} 6)${MENU} Restore ChromeOS Bitmaps ${NORMAL}"
	fi
	echo -e "${MENU}*********************************************************${NORMAL}"
	echo -e "${ENTER_LINE}Select a numeric menu option or${NORMAL}"
	echo -e "${nvram}${RED_TEXT}R${NORMAL} to reboot ${NORMAL} ${RED_TEXT}P${NORMAL} to poweroff ${NORMAL} ${RED_TEXT}Q${NORMAL} to quit ${NORMAL}"

	read -re opt
	case $opt in

		1)  if [[ "$unlockMenu" = true || ("$isChromeOS" = true && "$isCmlBook" = false \
					|| "$isFullRom" = false && "$isBootStub" = false && "$isUnsupported" = false) ]]; then
				flash_rwlegacy
			fi
			menu_fwupdate
			;;

		3)  if [[ "$unlockMenu" = true || "$isChromeOS" = true || "$isUnsupported" = false \
					&& "$isFullRom" = false && "$isBootStub" = false ]]; then
				set_boot_options
			fi
			menu_fwupdate
			;;

		4)  if [[ "$unlockMenu" = true || "$isChromeOS" = true || "$isUnsupported" = false \
					&& "$isFullRom" = false && "$isBootStub" = false ]]; then
				set_hwid
			fi
			menu_fwupdate
			;;

		5)  if [[ "$unlockMenu" = true || ( "$isFullRom" = false && "$isBootStub" = false && \
					( "$isHsw" = true || "$isBdw" = true || "$isByt" = true || "$isBsw" = true ) )  ]]; then
				remove_bitmaps
			fi
			menu_fwupdate
			;;

		6)  if [[ "$unlockMenu" = true || ( "$isFullRom" = false && "$isBootStub" = false && \
					( "$isHsw" = true || "$isBdw" = true || "$isByt" = true || "$isBsw" = true ) )  ]]; then
				restore_bitmaps
			fi
			menu_fwupdate
			;;

		[rR])  echo -e "\nRebooting...\n";
			cleanup
			reboot
			exit
			;;

		[pP])  echo -e "\nPowering off...\n";
			cleanup
			poweroff
			exit
			;;

		[qQ])  cleanup;
			exit;
			;;

		[U])  if [ "$unlockMenu" = false ]; then
				echo_yellow "\nAre you sure you wish to unlock all menu functions?"
				read -rep "Only do this if you really know what you are doing... [y/N]? "
				[[ "$REPLY" = "y" || "$REPLY" = "Y" ]] && unlockMenu=true
			fi
			menu_fwupdate
			;;

		*)  clear
			menu_fwupdate;
			;;
	esac
}
